"""
ReConf-Juggler aggregates dump tool.

"""
import argparse
import json
import logging
import sys

from infra.reconf import iterate_depth_first
from infra.reconf_juggler import resolvers
from infra.reconf_juggler.tools.jconv import html
from infra.reconf_juggler.util.jsdk import tree2jsdk


def set_loglevel(level_name):
    logging.getLogger().setLevel(level_name)

    return level_name


def main(resolver=None):
    logging.basicConfig(
        format='[%(asctime)s] %(levelname)s %(message)s',
        level=logging.DEBUG
    )

    argparser = argparse.ArgumentParser(description=__doc__)
    argparser.add_argument(
        '--namespace', '--ns',
        help='namespece of the aggregate',
        type=str,
    )
    argparser.add_argument(
        '--object-names', '--on',
        help='comma separated object names list',
        type=lambda x: sorted(set(x.split(','))),
    )
    argparser.add_argument(
        '--service-names', '--sn',
        help='comma separated service names list',
        type=lambda x: sorted(set(x.split(','))),
    )
    argparser.add_argument(
        '--tags',
        help='comma separated tags list',
        type=lambda x: sorted(set(x.split(','))),
    )
    argparser.add_argument(
        '--api-url',
        help='juggler api url; %(default)s is used by default',
        default='http://juggler-api.search.yandex.net'
    )
    argparser.add_argument(
        '--keys-exclude',
        help='comma separated keys to exclude from aggregates',
        default=(),
        type=lambda x: sorted(set(x.split(','))),
    )
    argparser.add_argument(
        '--keys-only',
        help='comma separated keys which should remain in aggregates',
        default=(),
        type=lambda x: set(x.split(',')),
    )
    argparser.add_argument(
        '--loglevel',
        choices=('CRITICAL', 'ERROR', 'WARNING', 'INFO', 'DEBUG'),
        default='INFO',
        type=set_loglevel,
    )
    argparser.add_argument(
        '--ofile',
        default=sys.stdout,
        type=argparse.FileType('w'),
        help='output file; STDOUT used by default',
    )
    argparser.add_argument(
        '--ofmt',
        help='output format; "%(default)s" used by default',
        choices=('html', 'jsdk', 'tree'),
        default='tree',
        type=str,
    )
    argparser.add_argument(
        '--recursive',
         help='dump recursively; "%(default)s" used by default',
         choices=(0, 1),
         default=0,
         type=int,
    )

    args = argparser.parse_args()

    if resolver is None:
        resolver = resolvers.RootResolver()

    dump = resolver['juggler']['aggregates'].resolve({
        'namespace_name': args.namespace,
        'object_names': args.object_names,
        'service_names': args.service_names,
        'tags': args.tags,
        'recursive': bool(args.recursive),
    })

    for name, body in iterate_depth_first(dump):
        if args.keys_only:  # empty filter means disabled
            for key in tuple(body.keys()):
                if key not in args.keys_only:
                    del body[key]
        else:
            for key in args.keys_exclude:
                if key in body:
                    del body[key]

    if args.ofmt == 'tree':
        json.dump(dump, args.ofile, indent=3, sort_keys=True)
    elif args.ofmt == 'html':
        args.ofile.write(html.tree2html(dump))
    elif args.ofmt == 'jsdk':
        json.dump(tree2jsdk(dump), args.ofile, indent=3, sort_keys=True)
    else:
        raise RuntimeError('Unsupported output format' + args.ofmt)
