import logging

import infra.reconf


class NodeTagsHandler(infra.reconf.OptHandler):
    @staticmethod
    def get_defaults():
        return {'tags': None}

    def get_default_value(self, key):
        val = self.get_parent_node_tags() + self._bound.get_tags()
        if val:
            return sorted(set(val))

        raise infra.reconf.ConfOptAbsent

    def get_handled_value(self, key, val):
        val = val + self.get_parent_node_tags()
        if val:
            return sorted(set(val))

        raise infra.reconf.ConfOptAbsent

    def get_parent_node_tags(self):
        try:
            return self._bound._bound._bound.get('tags', [])
        except AttributeError:
            return []


class NodeSet(infra.reconf.ConfSet):
    def __init__(self, *args, bound=None, **kwargs):
        if not self.branches:
            self.branches = (Node,)

        super().__init__(*args, bound=bound, **kwargs)

    def add_endpoint(self, name, body):
        if self.resolver is not None and self.resolve_endpoint(name) < 1:
            logging.debug('Discard endpoint (no instances) ' + name)
            return

        super().add_endpoint(name, body)

    def add_node(self, name, body):
        if 'children' in body and body['children']:
            super().add_node(name, body)
        else:
            logging.debug('Discard node (no children) ' + name)

    def get_endpoint_id(self, origin, node_class):
        if self._bound.selectors:
            merged = {}
            for level in self._bound.selectors:
                merged.update(level)

            suffix = '&'.join(k + '=' + v for k, v in sorted(merged.items()))
            return origin + '&' + suffix

        return origin

    def get_node_id(self, origin, node_class):
        return origin + node_class.get_name_suffix()

    def resolve_endpoint(self, name):
        return self.resolver['juggler']['instances_count'].resolve(name)


class NodeSetHandler(infra.reconf.SubnodesHandler):
    def get_handled_value(self, key, val):
        if self._bound.subnodes.branches:
            # add sublevel, so NodeSet can apply branches to it
            val = {self._bound.name: {'children': val}}
        else:
            # add current node name as prefix for subnodes origins
            new = {}
            for name, body in val.items():
                if body is not None and self._bound.name is not None:
                    name = self._bound.name + '_' + name
                new[name] = body

            val = new

        return super().get_handled_value(key, val)


class Node(infra.reconf.ConfNode):
    validate_class = False

    subnodes = NodeSet
    subnodes_handler = NodeSetHandler
    tags_handler = NodeTagsHandler

    @classmethod
    def get_selectors(cls):
        return None

    @classmethod
    def get_name_suffix(cls):
        return ''

    def get_tags(self):
        return []

    def init_subnodes(self):
        # tags required on subtree build stage
        self.add_handler(self.tags_handler, self.initial_dict)

        try:
            self.selectors = self._bound._bound.selectors
        except AttributeError:
            self.selectors = ()

        selector = self.get_selectors()
        if selector is not None:
            self.selectors = self.selectors + (selector,)

        super().init_subnodes()
