"""
Juggler sdk related stuff

"""
import infra.reconf


def jsdk2tree(checks):
    """
    Convert juggler-sdk checks list to CheckSet compatible tree structure.

    :param checks: juggler-sdk checks list

    """
    converted = {}
    relations = {}  # parents lookup index: check_id => [parents IDs list]

    # correct checks structure and build relations index
    for check in checks:
        check_id = check.pop('host') + ':' + check.pop('service')

        if check_id in converted:
            raise RuntimeError("Duplicate check found: " + check_id)

        if 'children' in check:
            children = {}

            for child in check['children']:
                child_id = child['host'] + ':' + child['service']

                if 'instance' in child and child['instance']:
                    child_id += ':' + child['instance']

                if 'group_type' in child and child['group_type'] != 'HOST':
                    child_id = child['group_type'] + '%' + child_id

                children[child_id] = None
                relations.setdefault(child_id, []).append(check_id)

            check['children'] = children

        converted[check_id] = check

    # link children to parents
    linked = set()
    for check_id, parents in relations.items():
        if check_id not in converted:
            continue

        for parent in parents:
            converted[parent]['children'][check_id] = converted[check_id]
            linked.add(check_id)

    # remove linked from root level
    for check_id in linked:
        del converted[check_id]

    return converted


def tree2jsdk(checks):
    """
    Convert CheckSet to juggler-sdk compatible list of checks.

    :param checks: reconf.juggler.CheckSet instance.

    """
    ret = []

    for name, body in infra.reconf.iterate_depth_first(checks):
        body = dict(body)
        body['host'], body['service'] = name.split(':')

        if 'children' in body:
            children = body.pop('children')
            body['children'] = []

            for child in sorted(children):
                converted = {}
                group_type = 'HOST'

                if children[child] is None:
                    try:
                        group_type, child = child.split('%')
                    except ValueError:  # no separator in name
                        pass

                    if group_type == 'EVENTS':
                        child = (child, 'all')
                    else:
                        child = child.split(':', 2)

                        try:
                            converted['instance'] = child[2]
                        except IndexError:
                            pass  # instance is optional
                else:
                    child = child.split(':', 1)

                converted['group_type'] = group_type
                converted['host'] = child[0]
                converted['service'] = child[1]

                body['children'].append(converted)

        ret.append(body)

    return ret
