#!/usr/bin/env python

import argparse
import bisect
import math
import json
import os.path


__version__ = '0.7.1'

FIELDS = ['INPUT', 'QPERC', 'QABST', 'QPRCI', 'QABSI', 'HPERC', 'HABST']

TABLES = {
    99: {'HPERC': {'11': 0.0481, '2': 0.2251, '21': 0.0257, '39': 0.0134, '5': 0.0993, '75': 0.006}, 'QPERC': {'16': 0.9375, '2': 0.5, '32': 0.9687, '4': 0.75, '64': 0.9843, '8': 0.875}},  # noqa E501
    98: {'HPERC': {'106': 0.006, '11': 0.0482, '2': 0.225, '21': 0.0268, '38': 0.0156, '5': 0.0983, '65': 0.0095}, 'QPERC': {'111': 0.9819, '16': 0.9375, '2': 0.5, '32': 0.9591, '4': 0.75, '74': 0.9729, '8': 0.875}},  # noqa E501
    97: {'HPERC': {'11': 0.0484, '112': 0.0082, '122': 0.0078, '2': 0.2251, '21': 0.0279, '36': 0.0181, '5': 0.0975, '56': 0.013, '77': 0.0104, '97': 0.009}, 'QPERC': {'124': 0.9729, '16': 0.92, '2': 0.5, '38': 0.9473, '4': 0.75, '57': 0.9649, '8': 0.875, '86': 0.9677}},  # noqa E501
    96: {'HPERC': {'11': 0.0485, '2': 0.2251, '21': 0.0289, '35': 0.0201, '5': 0.0966, '50': 0.0161, '63': 0.0141, '71': 0.0133, '76': 0.0128, '79': 0.0126}, 'QPERC': {'120': 0.9623, '16': 0.9047, '2': 0.5, '32': 0.9375, '4': 0.75, '48': 0.9482, '78': 0.9583, '8': 0.875}},  # noqa E501
    95: {'HPERC': {'11': 0.0486, '119': 0.0131, '2': 0.225, '21': 0.03, '34': 0.0221, '46': 0.0188, '5': 0.0956, '54': 0.0175, '58': 0.0169}, 'QPERC': {'118': 0.9516, '16': 0.8888, '2': 0.5, '27': 0.9259, '4': 0.75, '41': 0.9333, '60': 0.9436, '8': 0.875, '89': 0.9489}},  # noqa E501
    94: {'HPERC': {'11': 0.0488, '123': 0.0156, '2': 0.225, '21': 0.0311, '33': 0.0241, '42': 0.0216, '47': 0.0206, '49': 0.0203, '5': 0.0948, '99': 0.0163}, 'QPERC': {'116': 0.9406, '16': 0.8823, '2': 0.5, '26': 0.921, '4': 0.75, '51': 0.931, '73': 0.9367, '8': 0.875, '95': 0.9393}},  # noqa E501
    93: {'HPERC': {'104': 0.0187, '107': 0.0186, '11': 0.0489, '2': 0.2251, '21': 0.0321, '32': 0.0261, '39': 0.0241, '42': 0.0234, '43': 0.0232, '5': 0.0939, '87': 0.0193}, 'QPERC': {'113': 0.9304, '2': 0.5, '24': 0.9117, '4': 0.75, '46': 0.92, '63': 0.9253, '8': 0.875, '81': 0.9277, '97': 0.9292}},  # noqa E501
    92: {'HPERC': {'11': 0.0491, '2': 0.225, '21': 0.0332, '31': 0.028, '36': 0.0265, '38': 0.0261, '5': 0.0929, '77': 0.0221, '91': 0.0215}, 'QPERC': {'111': 0.9203, '2': 0.5, '24': 0.9032, '4': 0.75, '42': 0.9111, '57': 0.9152, '71': 0.9166, '8': 0.875, '84': 0.9186}},  # noqa E501
    91: {'HPERC': {'11': 0.0492, '124': 0.0234, '2': 0.225, '21': 0.0342, '30': 0.0298, '34': 0.0287, '35': 0.0284, '5': 0.092, '71': 0.0247, '81': 0.0243}, 'QPERC': {'110': 0.9097, '2': 0.5, '23': 0.8928, '38': 0.9024, '4': 0.75, '52': 0.9056, '64': 0.9076, '8': 0.8666, '87': 0.909}},  # noqa E501
    90: {'HPERC': {'100': 0.0263, '115': 0.0253, '119': 0.025, '13': 0.0662, '16': 0.0587, '18': 0.055, '2': 0.225, '37': 0.0382, '5': 0.1209, '53': 0.0328, '61': 0.031, '65': 0.0303, '9': 0.0828}, 'QPERC': {'109': 0.9026, '12': 0.7692, '124': 0.9051, '18': 0.826, '2': 0.5, '29': 0.8571, '4': 0.6, '42': 0.8723, '55': 0.8813, '68': 0.8888, '8': 0.75, '81': 0.8953, '96': 0.8989}},  # noqa E501
    89: {'HPERC': {'102': 0.0287, '105': 0.0285, '13': 0.0666, '16': 0.0594, '17': 0.0575, '2': 0.2251, '35': 0.0409, '49': 0.0358, '5': 0.12, '56': 0.0342, '59': 0.0336, '9': 0.0826, '90': 0.0296}, 'QPERC': {'108': 0.8918, '121': 0.8943, '18': 0.8181, '2': 0.5, '28': 0.8437, '39': 0.8604, '4': 0.6666, '51': 0.8703, '62': 0.8769, '74': 0.8815, '85': 0.8863, '9': 0.7692, '97': 0.8888}},  # noqa E501
    88: {'HPERC': {'126': 0.0299, '13': 0.067, '15': 0.0621, '17': 0.0583, '2': 0.2251, '35': 0.0425, '48': 0.0381, '5': 0.1191, '53': 0.0369, '82': 0.0328, '9': 0.0825, '92': 0.0319}, 'QPERC': {'108': 0.8818, '119': 0.8833, '18': 0.8095, '2': 0.5, '27': 0.8333, '36': 0.85, '4': 0.6666, '47': 0.8571, '56': 0.8644, '67': 0.8695, '77': 0.8734, '87': 0.8764, '9': 0.7692, '98': 0.8787}},  # noqa E501
    87: {'HPERC': {'115': 0.033, '122': 0.0327, '13': 0.0674, '15': 0.0627, '16': 0.0608, '2': 0.225, '33': 0.0451, '45': 0.0408, '5': 0.1182, '50': 0.0396, '76': 0.0357, '85': 0.0349, '86': 0.0348, '9': 0.0823}, 'QPERC': {'108': 0.8715, '117': 0.8728, '126': 0.8736, '18': 0.8095, '2': 0.5, '27': 0.8275, '35': 0.8378, '4': 0.6666, '44': 0.8478, '53': 0.8545, '62': 0.8593, '72': 0.863, '81': 0.8658, '9': 0.7692, '90': 0.8681, '99': 0.87}},  # noqa E501
    86: {'HPERC': {'107': 0.0359, '112': 0.0357, '13': 0.0678, '15': 0.0633, '16': 0.0615, '2': 0.2251, '33': 0.0467, '43': 0.0432, '47': 0.0422, '5': 0.1173, '72': 0.0384, '79': 0.0377, '9': 0.0822}, 'QPERC': {'107': 0.8611, '123': 0.8629, '18': 0.8, '2': 0.5, '25': 0.8214, '34': 0.8285, '4': 0.6666, '41': 0.8372, '50': 0.8431, '58': 0.8474, '66': 0.8507, '74': 0.8533, '82': 0.8554, '9': 0.7692, '90': 0.8586}},  # noqa E501
    85: {'HPERC': {'103': 0.0387, '13': 0.0682, '15': 0.0639, '16': 0.0622, '2': 0.2251, '33': 0.0483, '42': 0.0454, '5': 0.1164, '67': 0.0412, '73': 0.0406, '74': 0.0405, '9': 0.082, '99': 0.0389}, 'QPERC': {'114': 0.8521, '18': 0.7894, '2': 0.5, '24': 0.8148, '33': 0.8235, '4': 0.6666, '40': 0.8292, '47': 0.8333, '54': 0.8392, '69': 0.8428, '77': 0.8461, '9': 0.7692, '91': 0.8478, '99': 0.85}},  # noqa E501
    84: {'HPERC': {'121': 0.0405, '13': 0.0685, '15': 0.0645, '2': 0.225, '32': 0.0503, '40': 0.0477, '42': 0.0472, '5': 0.1155, '64': 0.0437, '69': 0.0432, '9': 0.0818, '93': 0.0416}, 'QPERC': {'106': 0.8407, '126': 0.842, '16': 0.7894, '2': 0.5, '24': 0.8076, '37': 0.8205, '4': 0.6666, '45': 0.826, '58': 0.8305, '65': 0.8333, '72': 0.8356, '9': 0.75, '92': 0.8387}},  # noqa E501
    83: {'HPERC': {'113': 0.0434, '13': 0.0689, '15': 0.0651, '2': 0.225, '31': 0.0522, '39': 0.0497, '5': 0.1146, '61': 0.0462, '65': 0.0458, '88': 0.0443, '9': 0.0817}, 'QPERC': {'118': 0.8314, '16': 0.7894, '2': 0.5, '24': 0.8, '30': 0.8064, '4': 0.6666, '42': 0.8181, '55': 0.8214, '68': 0.826, '9': 0.75, '92': 0.8297}},  # noqa E501
    82: {'HPERC': {'107': 0.0461, '13': 0.0693, '15': 0.0657, '2': 0.225, '31': 0.0538, '38': 0.0517, '5': 0.1137, '59': 0.0486, '62': 0.0483, '83': 0.047, '9': 0.0815}, 'QPERC': {'111': 0.8205, '16': 0.7777, '2': 0.5, '23': 0.7916, '29': 0.8, '35': 0.8055, '4': 0.6666, '47': 0.8113, '64': 0.8153, '76': 0.817, '9': 0.75, '93': 0.8191}},  # noqa E501
    81: {'HPERC': {'102': 0.0488, '123': 0.0483, '13': 0.0697, '15': 0.0663, '2': 0.2251, '31': 0.0553, '37': 0.0537, '5': 0.1128, '56': 0.051, '59': 0.0507, '79': 0.0495, '9': 0.0814}, 'QPERC': {'121': 0.8103, '16': 0.7777, '2': 0.5, '27': 0.7931, '34': 0.8, '4': 0.6666, '50': 0.8035, '72': 0.8076, '9': 0.75, '94': 0.8095}},  # noqa E501
    80: {'HPERC': {'1': 0.3159, '116': 0.0501, '120': 0.0499, '19': 0.0792, '26': 0.0711, '29': 0.0687, '4': 0.1583, '44': 0.061, '50': 0.0591, '51': 0.0588, '69': 0.055, '7': 0.1206, '73': 0.0543, '9': 0.1074, '93': 0.0519}, 'QPERC': {'1': 0.5, '105': 0.8018, '11': 0.6666, '111': 0.8035, '128': 0.8058, '15': 0.6875, '20': 0.7142, '25': 0.7407, '31': 0.75, '36': 0.7567, '42': 0.7674, '52': 0.7777, '59': 0.7833, '6': 0.625, '70': 0.7887, '76': 0.7922, '87': 0.7954, '93': 0.7978, '99': 0.8}},  # noqa E501
    79: {'HPERC': {'1': 0.3225, '110': 0.0531, '113': 0.0529, '19': 0.0801, '25': 0.0733, '28': 0.0708, '4': 0.1573, '43': 0.0632, '48': 0.0616, '65': 0.0578, '7': 0.12, '70': 0.057, '88': 0.0549, '9': 0.1073}, 'QPERC': {'1': 0.5, '11': 0.6666, '110': 0.7927, '121': 0.795, '15': 0.6875, '20': 0.7142, '25': 0.7307, '30': 0.7419, '35': 0.75, '40': 0.756, '50': 0.7692, '6': 0.625, '61': 0.7741, '67': 0.7794, '82': 0.7857, '94': 0.7894}},  # noqa E501
    78: {'HPERC': {'1': 0.3291, '105': 0.0559, '126': 0.0547, '128': 0.0546, '19': 0.0811, '25': 0.0746, '27': 0.073, '4': 0.1562, '42': 0.0654, '46': 0.0641, '63': 0.0604, '7': 0.1195, '83': 0.0578, '87': 0.0574, '9': 0.1071}, 'QPERC': {'1': 0.5, '11': 0.6666, '114': 0.7826, '120': 0.7847, '15': 0.6875, '20': 0.7142, '28': 0.7333, '34': 0.7428, '39': 0.75, '44': 0.7551, '58': 0.7627, '6': 0.625, '68': 0.7714, '79': 0.775, '89': 0.7777, '99': 0.78}},  # noqa E501
    77: {'HPERC': {'1': 0.3357, '100': 0.0588, '120': 0.0577, '19': 0.082, '25': 0.0759, '3': 0.1802, '40': 0.0679, '45': 0.0664, '6': 0.1276, '61': 0.0629, '64': 0.0624, '8': 0.1123, '81': 0.0604, '83': 0.0602, '9': 0.107}, 'QPERC': {'1': 0.5, '104': 0.7714, '11': 0.6666, '123': 0.7741, '15': 0.6875, '23': 0.72, '33': 0.7352, '42': 0.7441, '47': 0.75, '56': 0.7543, '6': 0.625, '66': 0.7605, '80': 0.7654, '94': 0.7684}},  # noqa E501
    76: {'HPERC': {'1': 0.3422, '114': 0.0605, '19': 0.0829, '25': 0.0771, '26': 0.0764, '3': 0.1792, '40': 0.0697, '58': 0.0655, '6': 0.1268, '77': 0.0631, '8': 0.112, '9': 0.1068, '96': 0.0615}, 'QPERC': {'1': 0.5, '104': 0.7614, '11': 0.6666, '122': 0.7637, '15': 0.6875, '23': 0.7083, '31': 0.7272, '41': 0.738, '54': 0.7454, '6': 0.625, '63': 0.75, '72': 0.7532, '90': 0.7582}},  # noqa E501
    75: {'HPERC': {'1': 0.3488, '109': 0.0633, '127': 0.0626, '19': 0.0839, '24': 0.0792, '3': 0.1783, '38': 0.0722, '42': 0.071, '57': 0.0678, '59': 0.0675, '6': 0.1261, '75': 0.0656, '8': 0.1117, '9': 0.1067, '92': 0.0643}, 'QPERC': {'1': 0.5, '108': 0.7521, '11': 0.6666, '15': 0.6875, '23': 0.7083, '31': 0.7187, '43': 0.7333, '6': 0.625, '60': 0.7419, '78': 0.7468, '95': 0.75}},  # noqa E501
    74: {'HPERC': {'105': 0.0661, '106': 0.066, '122': 0.0654, '19': 0.0848, '2': 0.225, '24': 0.0804, '25': 0.0797, '38': 0.0739, '41': 0.0731, '5': 0.1363, '55': 0.0702, '57': 0.07, '72': 0.0682, '8': 0.1114, '88': 0.067, '9': 0.1065}, 'QPERC': {'11': 0.6666, '115': 0.7413, '120': 0.7426, '18': 0.6842, '2': 0.5, '26': 0.7037, '34': 0.7142, '42': 0.7209, '54': 0.7272, '6': 0.625, '66': 0.7313, '71': 0.7361, '91': 0.7391}},  # noqa E501
    73: {'HPERC': {'101': 0.0688, '117': 0.0682, '19': 0.0857, '2': 0.2251, '24': 0.0816, '37': 0.076, '5': 0.1354, '53': 0.0726, '69': 0.0708, '8': 0.111, '85': 0.0696}, 'QPERC': {'11': 0.6666, '115': 0.7317, '18': 0.6842, '2': 0.5, '26': 0.7, '37': 0.7105, '45': 0.7173, '6': 0.625, '64': 0.723, '76': 0.7272, '99': 0.73}},  # noqa E501
    72: {'HPERC': {'127': 0.0705, '19': 0.0867, '2': 0.225, '24': 0.0829, '37': 0.0777, '5': 0.1345, '52': 0.0748, '67': 0.0732, '8': 0.1107, '82': 0.0721, '98': 0.0714}, 'QPERC': {'10': 0.6666, '103': 0.7207, '18': 0.6818, '2': 0.5, '29': 0.7, '4': 0.6, '40': 0.7073, '55': 0.7118, '66': 0.7142, '77': 0.7176}},  # noqa E501
    71: {'HPERC': {'122': 0.0732, '19': 0.0876, '2': 0.225, '23': 0.0847, '36': 0.0796, '5': 0.1336, '51': 0.077, '52': 0.0768, '66': 0.0755, '8': 0.1104, '80': 0.0746, '94': 0.074}, 'QPERC': {'10': 0.6666, '128': 0.7111, '18': 0.6818, '2': 0.5, '32': 0.6944, '4': 0.6, '46': 0.7, '57': 0.7049, '78': 0.7073, '96': 0.71}},  # noqa E501
    70: {'HPERC': {'1': 0.2699, '104': 0.0759, '119': 0.0748, '14': 0.1147, '18': 0.1067, '19': 0.1051, '29': 0.0945, '32': 0.0924, '4': 0.1763, '44': 0.0866, '58': 0.0824, '6': 0.1521, '61': 0.0817, '74': 0.0794, '89': 0.0774}, 'QPERC': {'1': 0.5, '103': 0.7019, '118': 0.7058, '16': 0.6111, '26': 0.6428, '34': 0.6571, '41': 0.6666, '48': 0.673, '59': 0.6825, '70': 0.6891, '8': 0.5454, '81': 0.6941, '92': 0.6979}},  # noqa E501
    69: {'HPERC': {'1': 0.2765, '100': 0.0788, '115': 0.0777, '14': 0.1152, '18': 0.1075, '28': 0.0967, '4': 0.1752, '42': 0.0892, '45': 0.0881, '57': 0.0848, '59': 0.0843, '6': 0.1514, '72': 0.082, '86': 0.0802}, 'QPERC': {'1': 0.5, '110': 0.6929, '121': 0.6958, '16': 0.5882, '20': 0.619, '29': 0.6333, '33': 0.647, '40': 0.6585, '50': 0.6666, '57': 0.6721, '71': 0.6799, '8': 0.5454, '85': 0.6853, '96': 0.69}},  # noqa E501
    68: {'HPERC': {'1': 0.2831, '111': 0.0805, '125': 0.0797, '14': 0.1157, '18': 0.1084, '28': 0.0981, '31': 0.0961, '4': 0.1742, '42': 0.091, '44': 0.0903, '56': 0.0871, '6': 0.1507, '69': 0.0847, '71': 0.0844, '83': 0.0829, '97': 0.0816}, 'QPERC': {'1': 0.5, '103': 0.6818, '120': 0.6853, '16': 0.5882, '22': 0.6086, '26': 0.6296, '38': 0.65, '52': 0.6603, '59': 0.6666, '69': 0.671, '8': 0.5454, '86': 0.6774}},  # noqa E501
    67: {'HPERC': {'1': 0.2896, '107': 0.0833, '121': 0.0825, '15': 0.1142, '18': 0.1092, '28': 0.0995, '4': 0.1731, '41': 0.0932, '43': 0.0925, '55': 0.0894, '56': 0.0892, '6': 0.15, '68': 0.0871, '7': 0.1425, '81': 0.0855, '94': 0.0843, '95': 0.0842}, 'QPERC': {'1': 0.5, '116': 0.6746, '16': 0.5882, '22': 0.6086, '31': 0.625, '35': 0.6388, '50': 0.647, '54': 0.6545, '67': 0.6617, '8': 0.5454, '80': 0.6666, '93': 0.6699}},  # noqa E501
    66: {'HPERC': {'1': 0.2962, '104': 0.086, '117': 0.0853, '15': 0.1148, '18': 0.1101, '28': 0.1009, '30': 0.0997, '4': 0.1721, '41': 0.095, '53': 0.0919, '6': 0.1493, '66': 0.0897, '67': 0.0895, '7': 0.142, '79': 0.0881, '91': 0.087, '92': 0.0869}, 'QPERC': {'1': 0.5, '109': 0.6636, '128': 0.6644, '16': 0.5882, '22': 0.6086, '31': 0.625, '40': 0.6341, '52': 0.6415, '62': 0.6507, '8': 0.5454, '86': 0.659}},  # noqa E501
    65: {'HPERC': {'1': 0.3028, '101': 0.0887, '113': 0.0881, '125': 0.0875, '15': 0.1154, '18': 0.111, '28': 0.1024, '4': 0.1711, '40': 0.0972, '52': 0.0942, '6': 0.1486, '64': 0.0922, '7': 0.1415, '76': 0.0907, '89': 0.0896}, 'QPERC': {'1': 0.5, '114': 0.6521, '127': 0.6537, '16': 0.5882, '22': 0.6, '30': 0.6129, '39': 0.625, '48': 0.6326, '60': 0.6393, '75': 0.6447, '8': 0.5454, '93': 0.6489}},  # noqa E501
    64: {'HPERC': {'1': 0.3093, '110': 0.0908, '15': 0.116, '26': 0.1049, '29': 0.1032, '39': 0.0993, '4': 0.17, '51': 0.0964, '6': 0.1479, '63': 0.0945, '7': 0.1409, '75': 0.0931, '86': 0.0922, '98': 0.0914}, 'QPERC': {'1': 0.5, '105': 0.6415, '120': 0.6432, '16': 0.5789, '24': 0.6, '30': 0.6111, '44': 0.6222, '53': 0.6271, '70': 0.6338, '8': 0.5454, '82': 0.6373}},  # noqa E501
    63: {'HPERC': {'1': 0.3159, '117': 0.093, '15': 0.1166, '26': 0.1063, '38': 0.1013, '4': 0.169, '50': 0.0986, '6': 0.1472, '61': 0.0969, '7': 0.1404, '73': 0.0956, '84': 0.0948, '95': 0.0941}, 'QPERC': {'1': 0.5, '108': 0.6315, '125': 0.6327, '16': 0.5789, '24': 0.5925, '32': 0.606, '46': 0.6153, '60': 0.6212, '74': 0.625, '8': 0.5454, '88': 0.6288}},  # noqa E501
    62: {'HPERC': {'1': 0.3224, '104': 0.0961, '125': 0.0954, '15': 0.1172, '26': 0.1076, '28': 0.1066, '38': 0.1031, '4': 0.168, '49': 0.1008, '6': 0.1464, '60': 0.0992, '7': 0.1399, '71': 0.0981, '82': 0.0973, '93': 0.0966}, 'QPERC': {'1': 0.5, '105': 0.6203, '116': 0.6218, '127': 0.6221, '18': 0.5789, '29': 0.5937, '37': 0.6, '45': 0.6078, '64': 0.6142, '8': 0.5454, '83': 0.6179}},  # noqa E501
    61: {'HPERC': {'1': 0.329, '101': 0.0988, '122': 0.0981, '15': 0.1178, '17': 0.1154, '26': 0.1089, '37': 0.1051, '4': 0.1669, '48': 0.1029, '59': 0.1015, '6': 0.1457, '69': 0.1005, '7': 0.1394, '80': 0.0998}, 'QPERC': {'1': 0.5, '118': 0.6115, '18': 0.5789, '31': 0.5937, '47': 0.6, '60': 0.6031, '76': 0.6071, '8': 0.5454, '97': 0.61}},  # noqa E501
    60: {'HPERC': {'1': 0.2239, '109': 0.1005, '11': 0.1489, '120': 0.0996, '14': 0.1404, '22': 0.127, '24': 0.1248, '33': 0.1175, '43': 0.1125, '5': 0.1833, '54': 0.1089, '65': 0.1063, '66': 0.106, '76': 0.1043, '87': 0.1028, '98': 0.1015}, 'QPERC': {'1': 0.5045, '102': 0.6018, '116': 0.605, '127': 0.607, '14': 0.5, '2': 0.3333, '20': 0.5217, '30': 0.5454, '40': 0.5609, '48': 0.5714, '56': 0.5789, '6': 0.4285, '67': 0.5857, '75': 0.5897, '83': 0.5952, '94': 0.5979}},  # noqa E501
    59: {'HPERC': {'1': 0.2305, '106': 0.1033, '11': 0.149, '117': 0.1024, '127': 0.1018, '14': 0.1409, '22': 0.1281, '32': 0.1197, '42': 0.1148, '5': 0.1824, '53': 0.1112, '54': 0.1109, '64': 0.1086, '74': 0.1069, '85': 0.1054, '95': 0.1043}, 'QPERC': {'1': 0.4783, '105': 0.5925, '118': 0.595, '126': 0.5965, '14': 0.5, '22': 0.52, '3': 0.3333, '32': 0.5428, '42': 0.5555, '50': 0.566, '6': 0.4285, '60': 0.5714, '68': 0.5774, '76': 0.5822, '89': 0.5869, '97': 0.59}},  # noqa E501
    58: {'HPERC': {'1': 0.237, '103': 0.106, '114': 0.1052, '12': 0.1463, '124': 0.1046, '14': 0.1414, '22': 0.1292, '31': 0.1219, '33': 0.1207, '42': 0.1166, '5': 0.1815, '52': 0.1135, '53': 0.1132, '62': 0.1112, '63': 0.111, '73': 0.1094, '83': 0.108, '93': 0.1069}, 'QPERC': {'1': 0.4521, '107': 0.5833, '120': 0.5853, '128': 0.586, '14': 0.5, '22': 0.5185, '3': 0.3333, '34': 0.5405, '44': 0.5531, '54': 0.5614, '6': 0.4285, '64': 0.5671, '74': 0.5714, '84': 0.5747, '92': 0.5789}},  # noqa E501
    57: {'HPERC': {'1': 0.2436, '101': 0.1087, '111': 0.108, '12': 0.1465, '121': 0.1074, '14': 0.1419, '22': 0.1303, '31': 0.1234, '41': 0.1188, '5': 0.1806, '51': 0.1158, '61': 0.1136, '71': 0.112, '81': 0.1107, '91': 0.1096}, 'QPERC': {'1': 0.4257, '102': 0.5714, '112': 0.5739, '127': 0.5754, '14': 0.5, '24': 0.5172, '3': 0.3333, '38': 0.5365, '48': 0.5471, '6': 0.4285, '60': 0.5555, '70': 0.5599, '82': 0.5647, '92': 0.5684}},  # noqa E501
    56: {'HPERC': {'1': 0.2501, '107': 0.1108, '11': 0.1495, '118': 0.1101, '14': 0.1424, '22': 0.1314, '31': 0.125, '32': 0.1244, '4': 0.1901, '41': 0.1207, '50': 0.118, '60': 0.1159, '78': 0.1133, '89': 0.1122}, 'QPERC': {'1': 0.3996, '102': 0.5619, '119': 0.5645, '14': 0.4666, '17': 0.5, '26': 0.5161, '3': 0.3333, '40': 0.5333, '54': 0.5438, '6': 0.4285, '66': 0.5492, '78': 0.5542, '90': 0.5578}},  # noqa E501
    55: {'HPERC': {'1': 0.2567, '11': 0.1496, '114': 0.1129, '21': 0.1335, '30': 0.127, '4': 0.189, '40': 0.1228, '49': 0.1203, '59': 0.1183, '68': 0.1169, '77': 0.1158, '87': 0.1148, '96': 0.1141}, 'QPERC': {'1': 0.3732, '102': 0.5514, '116': 0.5537, '14': 0.4666, '17': 0.5, '28': 0.5142, '3': 0.3333, '44': 0.5294, '6': 0.4285, '60': 0.5384, '74': 0.5443, '88': 0.5483}},  # noqa E501
    54: {'HPERC': {'1': 0.2633, '103': 0.1161, '11': 0.1498, '112': 0.1156, '21': 0.1346, '30': 0.1285, '39': 0.1249, '4': 0.188, '49': 0.1223, '58': 0.1206, '67': 0.1193, '76': 0.1183, '85': 0.1174, '94': 0.1167}, 'QPERC': {'1': 0.347, '102': 0.5412, '120': 0.5438, '14': 0.4666, '19': 0.5, '3': 0.3333, '32': 0.5128, '52': 0.5254, '6': 0.4285, '68': 0.5333, '86': 0.5376}},  # noqa E501
    53: {'HPERC': {'1': 0.2699, '101': 0.1187, '11': 0.1499, '118': 0.1179, '21': 0.1356, '30': 0.13, '39': 0.1267, '4': 0.187, '48': 0.1245, '57': 0.1228, '74': 0.1208, '83': 0.12, '92': 0.1193}, 'QPERC': {'1': 0.3333, '102': 0.5315, '124': 0.5334, '14': 0.4666, '21': 0.5, '36': 0.5106, '6': 0.4285, '60': 0.5217, '82': 0.5274}},  # noqa E501
    52: {'HPERC': {'1': 0.2764, '107': 0.121, '11': 0.15, '116': 0.1206, '20': 0.1375, '22': 0.1359, '30': 0.1315, '4': 0.1859, '46': 0.1268, '64': 0.1241, '73': 0.1232, '90': 0.1219}, 'QPERC': {'1': 0.3333, '104': 0.5217, '14': 0.4666, '25': 0.5, '44': 0.5084, '6': 0.4285, '76': 0.5172}},  # noqa E501
    51: {'HPERC': {'1': 0.283, '105': 0.1236, '11': 0.1502, '122': 0.1229, '20': 0.1385, '29': 0.1334, '30': 0.133, '38': 0.1305, '4': 0.1849, '46': 0.1288, '63': 0.1264, '80': 0.1249}, 'QPERC': {'1': 0.4, '10': 0.4545, '106': 0.5121, '22': 0.4782, '35': 0.5, '62': 0.5061}},  # noqa E501
    50: {'HPERC': {'1': 0.204, '10': 0.1776, '103': 0.126, '120': 0.1245, '17': 0.16, '19': 0.1568, '26': 0.1486, '27': 0.1478, '34': 0.1427, '43': 0.1381, '51': 0.1352, '60': 0.1327, '68': 0.1309, '77': 0.1293, '93': 0.1271}, 'QPERC': {'1': 0.6887, '104': 0.5043, '12': 0.3846, '18': 0.4, '2': 0.3333, '25': 0.423, '33': 0.4545, '44': 0.4666, '59': 0.4833, '76': 0.4935, '89': 0.5}},  # noqa E501
}


class Calculator():
    def __init__(self, init_builtin_tables=False):
        self.__idx = {}  # indexes for tables
        self.__tbl = {}  # loaded tables

        if init_builtin_tables:
            for name, table in TABLES.items():
                self.set_table(name, TABLES[name])

    @staticmethod
    def _binsearch(array, value):
        i = bisect.bisect_right(array, value)
        return array[i - 1] if i else array[0]

    def get_table(self, name):
        if name in self.__tbl:
            return self.__tbl[name]

        return None

    def calculate(self, value, table_name):
        quorum = {}
        quorum['INPUT'] = value

        h_key = self._binsearch(self.__idx[table_name]['HPERC'], value)
        quorum['HPERC'] = self.__tbl[table_name]['HPERC'][h_key]        # hysteresis, percents
        quorum['HABST'] = int(value * quorum['HPERC'])                  # hysteresis, absolute

        q_key = self._binsearch(self.__idx[table_name]['QPERC'], value)
        quorum['QPERC'] = self.__tbl[table_name]['QPERC'][q_key]        # quorum, percents
        quorum['QPRCI'] = round(1 - quorum['QPERC'], 4)                 # quorum, inverted percents
        quorum['QABST'] = int(math.ceil(value * quorum['QPERC']))       # quorum, absolute
        quorum['QABSI'] = value - quorum['QABST']                       # quorum, inverted absolute

        return quorum

    def set_table(self, name, table):
        self.__tbl[name] = {}
        self.__idx[name] = {}
        for ptype in ('HPERC', 'QPERC'):
            self.__tbl[name][ptype] = {}
            for val in table[ptype]:
                self.__tbl[name][ptype][int(val)] = float(table[ptype][val])

            self.__idx[name][ptype] = sorted(self.__tbl[name][ptype])

        return self.__tbl[name]


def main():
    parser = argparse.ArgumentParser(description='Calculate quorum and hysteresis for given values')
    parser.add_argument('values', metavar='int', type=int, nargs='*', help='input values')
    parser.add_argument('-d', '--dump', action='store_true', help='calculate nothing, just dump table')
    parser.add_argument('-f', '--fields', metavar='fields', type=str, help='comma separated fields to show')
    parser.add_argument('-j', '--json', action='store_true', help='use JSON as output format')
    parser.add_argument('-t', '--table', metavar='name', type=str, help='quorum table to use')
    parser.add_argument('-q', '--quiet', action='store_true', help='suppress headers')
    parser.add_argument('-V', '--version', action='version', version='%(prog)s {}'.format(__version__),
                        help='print version and exit')
    args = parser.parse_args()

    if args.table:
        if os.path.exists(args.table):
            with open(args.table) as f:
                table = json.load(f)
        else:
            try:
                table = TABLES[int(args.table)]
            except KeyError:
                raise RuntimeError('No such table')
    else:
        table = TABLES[90]

    if args.dump:
        print(json.dumps(table, indent=3, sort_keys=True))
        raise SystemExit(0)

    if not args.values:
        raise RuntimeError('Too few arguments')

    fields = []
    if args.fields:
        for f in args.fields.split(','):
            if f not in FIELDS:
                raise RuntimeError('Unsupported field: ' + f)
            fields.append(f)
    else:
        fields = FIELDS

    result = {}
    calc = Calculator()
    calc.set_table('default', table)

    if not args.json and not args.quiet:
        print('#' + '\t'.join(fields))

    for value in args.values:
        if args.json:
            result[value] = calc.calculate(value, 'default')
        else:
            result = calc.calculate(value, 'default')
            print('\t'.join(str(result[f]) for f in fields))

    if args.json:
        print(json.dumps(result, indent=3, sort_keys=True))
