from __future__ import unicode_literals
import logging
import os

import flask
import gevent
import yp.client
from sepelib.core import config

from infra.swatlib import metrics
from infra.swatlib import webserver
from infra.swatlib.gevent import exclusiveservice
from infra.swatlib.zk import client as zk_client
from infra.release_controller.src import controller
from infra.release_controller.src import deploy_ticket_maker
from infra.release_controller.src import release_matcher
from infra.release_controller.src import release_selector
from infra.release_controller.src import processor
from infra.release_controller.src import storage
from infra.release_controller.src.lib import yp_client


class Application(object):
    name = 'release_controller'

    def __init__(self, instance_id):
        self.instance_id = instance_id
        self.log = logging.getLogger(self.name)

    @staticmethod
    def setup_environment():
        # Patch requests connection pool to use gevent queue
        from requests.packages.urllib3.connectionpool import ConnectionPool
        from gevent.queue import LifoQueue

        ConnectionPool.QueueCls = LifoQueue
        # Disable requests spamming about:
        # Connection pool is full, discarding connection
        # This is the way we use alemate-http to avoid broken connections
        # There is nothing we can do about it, so simply mute
        logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.ERROR)
        logging.getLogger('ydb.connection').setLevel(logging.ERROR)

    @staticmethod
    def make_coord(config, instance_id):
        return zk_client.ZookeeperClient(
            cfg={
                'hosts': config.get_value('coord.hosts'),
                'zk_root': config.get_value('coord.root'),
                'read_only': False,
                'log_debug': config.get_value('coord.log_debug'),
            },
            identifier=instance_id,
        )

    @staticmethod
    def make_yp_client(config):
        token = config.get('token')
        if not token:
            config['token'] = os.environ.get('YP_TOKEN')
        base = yp.client.YpClient(address=config['address'], config=config)
        stub = base.create_grpc_object_stub()
        return yp_client.YpClient(stub=stub)

    def init_http_server(self, config, metrics_registry):
        app = flask.Flask('release_controller')
        app.add_url_rule('/ping', view_func=lambda: '')
        return webserver.WebServer(config.get(), app, version='0.0.1', metrics_registry=metrics_registry)

    def run(self):
        self.setup_environment()

        # Init coordination service zk_client
        # Several instances can run on one host, use port for distinction.
        self.coord = self.make_coord(config, self.instance_id)
        client = self.make_yp_client(config.get_value('yp'))

        # Make storages for release rules and stages.
        release_rule_storage = storage.make_storage(storage.RELEASE_RULE_INDEXERS)
        stage_storage = storage.make_storage(storage.STAGE_INDEXERS)

        # Make release matcher.
        match_sandbox_resource_attributes = config.get_value('release_matcher.match_sandbox_resource_attributes', False)
        matcher = release_matcher.ReleaseMatcher(
            release_rule_storage=release_rule_storage,
            stage_storage=stage_storage,
            match_sandbox_resource_attributes=match_sandbox_resource_attributes
        )

        # Make deploy ticket maker.
        ticket_maker = deploy_ticket_maker.DeployTicketMaker()

        # Make release controller.
        ctl = controller.ReleaseController(yp_client=client,
                                           release_matcher=matcher,
                                           deploy_ticket_maker=ticket_maker,
                                           release_rule_storage=release_rule_storage,
                                           stage_storage=stage_storage)

        # Make release selector.
        selector = release_selector.NotProcessedReleaseBatchSelector(yp_client=client)
        reg = metrics.Registry()
        # Make release processor.
        proc = processor.ReleaseProcessor(release_selector=selector,
                                          release_controller=ctl,
                                          metrics_registry=reg)
        # self.processor = proc
        self.processor = exclusiveservice.ExclusiveService(self.coord, 'release_ctl', proc,
                                                           exit_instead_of_release_lock=True)

        self.log.info('Init and starting http server...')
        # Instantiate http server for stats.
        self.srv = self.init_http_server(config, metrics_registry=reg)
        # Run http server in separate thread.
        gevent.spawn(self.srv.run)

        self.log.info('Starting release processor...')
        self.coord.start().wait()
        self.processor.start()
        self.processor.wait()

    def stop(self):
        self.log.info('Stopping release processor...')
        self.processor.stop()
        self.log.info('Stopping http server...')
        self.srv.stop()
        self.log.info('Done.')
