from __future__ import unicode_literals
import logging
import time

import gevent

import yp.common
import yp.data_model
from infra.release_controller.src import storage
from infra.release_controller.src.lib import pbutil


class ReleaseController(object):
    SUCCESS_PROCESSING_TEMPLATE = ('Release processed successfully. '
                                   'Created tickets: {}.')
    RELEASE_RULE_CACHE_TTL = 20

    def __init__(self, yp_client, release_matcher, deploy_ticket_maker,
                 release_rule_storage, stage_storage, cache_ttl=None):
        self.yp_client = yp_client
        self.release_matcher = release_matcher
        self.deploy_ticket_maker = deploy_ticket_maker
        self.release_rule_storage = release_rule_storage
        self.stage_storage = stage_storage
        self.cache_ttl = cache_ttl or self.RELEASE_RULE_CACHE_TTL
        self._sync_time = None
        self.log = logging.getLogger(__name__)

    def create_deploy_ticket(self, ticket):
        try:
            self.yp_client.create_deploy_ticket(ticket)
            return True
        except yp.common.YpDuplicateObjectIdError:
            self.log.warning('Deploy ticket %s already exists for release %s', ticket.meta.id, ticket.spec.release_id)
            return False

    # TODO: let's think about moving resync_* methods to StorageResyncer
    def _resync_with_batches(self, s, batches):
        synced = storage.make_storage(s.indexers)
        for batch in batches:
            for obj in batch:
                synced.add(obj)
            gevent.sleep(0)
        s.update_from_other_storage(synced)

    def resync_release_rule_storage(self):
        ts = self.yp_client.generate_timestamp()
        batches = self.yp_client.select_release_rule_batches(timestamp=ts)
        return self._resync_with_batches(self.release_rule_storage, batches)

    def resync_stage_storage(self):
        ts = self.yp_client.generate_timestamp()
        batches = self.yp_client.select_stage_batches(timestamp=ts)
        return self._resync_with_batches(self.stage_storage, batches)

    def resync_storages_if_needed(self):
        is_actual = (self._sync_time is not None and
                     time.time() < self._sync_time + self.cache_ttl)
        if is_actual:
            return self.release_rule_storage.count(), self.stage_storage.count()
        sync_time = time.time()
        self.resync_release_rule_storage()
        self.resync_stage_storage()
        self._sync_time = sync_time
        return self.release_rule_storage.count(), self.stage_storage.count()

    def process(self, release):
        self.log.info('Starting process release: %s', release.meta.id)
        created_tickets = 0
        try:
            matched_rules = self.release_matcher.match(release)
            self.log.info('Found %d matched rules for release: %s', len(matched_rules), release.meta.id)
            for rule in matched_rules:
                stage = self.stage_storage.get(rule.meta.stage_id)
                if not stage:
                    self.log.error('Stage %s not found for release rule %s. Ticket will not be created',
                                   rule.meta.stage_id, rule.meta.id)
                    continue
                t = self.deploy_ticket_maker.make_deploy_ticket(release, rule, stage)
                created = self.create_deploy_ticket(t)
                if created:
                    created_tickets += 1
        except Exception as e:
            self.log.exception('Release %s processing failed', release.meta.id)
            pbutil.set_condition_exception(release.status.processing.finished, e)
        else:
            self.log.info('Release %s processing finished successfully. Created tickets: %d', release.meta.id, created_tickets)
            pbutil.set_condition_success(release.status.processing.finished,
                                         self.SUCCESS_PROCESSING_TEMPLATE.format(len(matched_rules)))
        self.yp_client.update_release_status_processing(release_id=release.meta.id,
                                                        processing=release.status.processing)
        return created_tickets
