from __future__ import unicode_literals

from gevent import threadpool

import yp.data_model
import yt_yson_bindings
from yp_proto.yp.client.api.proto import object_service_pb2

from infra.release_controller.src.lib import yputil


class YpClient(object):

    DEFAULT_BATCH_SIZE = 100

    def __init__(self, stub):
        self.stub = stub
        self._tp = threadpool.ThreadPool(maxsize=10)

    def _get_objects(self, object_ids, object_type, object_class,
                     selectors=None, timestamp=None):
        req = object_service_pb2.TReqGetObjects()
        if timestamp is not None:
            req.timestamp = timestamp
        selectors = selectors or ['']
        req.selector.paths.extend(selectors)
        req.object_type = object_type
        for object_id in object_ids:
            subreq = req.subrequests.add()
            subreq.object_id = object_id
        req.format = object_service_pb2.PF_YSON
        resp = self._tp.apply(self.stub.GetObjects, [req])
        paths = yputil.YSON_LOADER.make_attr_paths(selectors)
        rv = []
        for subresp in resp.subresponses:
            obj = yputil.YSON_LOADER.load_object_attrs(object_class,
                                                       paths,
                                                       subresp.result.value_payloads)
            rv.append(obj)
        return rv

    def _select_objects(self, object_type, object_class,
                        limit, continuation_token=None,
                        query=None, selectors=None, timestamp=None):
        req = object_service_pb2.TReqSelectObjects()
        if query:
            req.filter.query = query
        selectors = selectors or ['']
        req.selector.paths.extend(selectors)
        if timestamp is not None:
            req.timestamp = timestamp
        req.object_type = object_type
        req.limit.value = limit
        if continuation_token is not None:
            req.options.continuation_token = continuation_token
        req.format = object_service_pb2.PF_YSON
        resp = self._tp.apply(self.stub.SelectObjects, [req])
        paths = yputil.YSON_LOADER.make_attr_paths(selectors)
        rv = []
        for r in resp.results:
            obj = yputil.YSON_LOADER.load_object_attrs(object_class,
                                                       paths,
                                                       r.value_payloads)
            rv.append(obj)
        return rv, resp.continuation_token

    def _select_batches(self, object_type, object_class, query,
                        batch_size=None, batches_count=None, selectors=None,
                        timestamp=None):
        rv = []
        batch_size = batch_size or self.DEFAULT_BATCH_SIZE
        continuation_token = None
        while True:
            if batches_count is not None and len(rv) == batches_count:
                break
            batch, continuation_token = self._select_objects(object_type=object_type,
                                                             object_class=object_class,
                                                             continuation_token=continuation_token,
                                                             limit=batch_size,
                                                             query=query,
                                                             selectors=selectors,
                                                             timestamp=timestamp)
            if batch:
                rv.append(batch)
            if len(batch) < batch_size:
                break
        return rv

    def _create_object(self, object_pb, object_type):
        req = object_service_pb2.TReqCreateObject()
        req.object_type = object_type
        req.attributes = yt_yson_bindings.dumps_proto(object_pb)
        rsp = self._tp.apply(self.stub.CreateObject, [req])
        return rsp.object_id

    def _update_object(self, object_id, object_type, updates_yson):
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = object_type
        req.object_id = object_id
        for path, v in updates_yson.iteritems():
            upd = req.set_updates.add()
            upd.path = path
            upd.value = v
        self._tp.apply(self.stub.UpdateObject, [req])

    def generate_timestamp(self):
        req = object_service_pb2.TReqGenerateTimestamp()
        rsp = self.stub.GenerateTimestamp(req)
        return rsp.timestamp

    def get_releases(self, release_ids, selectors=None, timestamp=None):
        return self._get_objects(object_ids=release_ids,
                                 object_type=yp.data_model.OT_RELEASE,
                                 object_class=yp.data_model.TRelease,
                                 selectors=selectors,
                                 timestamp=timestamp)

    def select_releases(self, limit,
                        query=None, selectors=None, timestamp=None):
        releases, _ = self._select_objects(object_type=yp.data_model.OT_RELEASE,
                                           object_class=yp.data_model.TRelease,
                                           limit=limit,
                                           query=query,
                                           selectors=selectors,
                                           timestamp=timestamp)
        return releases

    def select_release_batches(self,
                               batch_size=None,
                               batches_count=None,
                               query=None,
                               selectors=None,
                               timestamp=None):
        return self._select_batches(object_type=yp.data_model.OT_RELEASE,
                                    object_class=yp.data_model.TRelease,
                                    query=query,
                                    batch_size=batch_size,
                                    batches_count=batches_count,
                                    selectors=selectors,
                                    timestamp=timestamp)

    def select_release_rule_batches(self,
                                    batch_size=None,
                                    batches_count=None,
                                    query=None,
                                    selectors=None,
                                    timestamp=None):
        return self._select_batches(object_type=yp.data_model.OT_RELEASE_RULE,
                                    object_class=yp.data_model.TReleaseRule,
                                    query=query,
                                    batch_size=batch_size,
                                    batches_count=batches_count,
                                    selectors=selectors,
                                    timestamp=timestamp)

    def select_stage_batches(self,
                             batch_size=None,
                             batches_count=None,
                             query=None,
                             selectors=None,
                             timestamp=None):
        return self._select_batches(object_type=yp.data_model.OT_STAGE,
                                    object_class=yp.data_model.TStage,
                                    query=query,
                                    batch_size=batch_size,
                                    batches_count=batches_count,
                                    selectors=selectors,
                                    timestamp=timestamp)

    def select_deploy_ticket_batches(self,
                                     batch_size=None,
                                     batches_count=None,
                                     query=None,
                                     selectors=None,
                                     timestamp=None):
        return self._select_batches(object_type=yp.data_model.OT_DEPLOY_TICKET,
                                    object_class=yp.data_model.TDeployTicket,
                                    query=query,
                                    batch_size=batch_size,
                                    batches_count=batches_count,
                                    selectors=selectors,
                                    timestamp=timestamp)

    def create_deploy_ticket(self, deploy_ticket):
        return self._create_object(object_pb=deploy_ticket,
                                   object_type=yp.data_model.OT_DEPLOY_TICKET)

    def update_release_status_processing(self, release_id, processing):
        updates_yson = {'/status/processing': yt_yson_bindings.dumps_proto(processing)}
        self._update_object(object_id=release_id,
                            object_type=yp.data_model.OT_RELEASE,
                            updates_yson=updates_yson)
