from __future__ import unicode_literals

import logging

import gevent

from infra.swatlib.gevent import greenthread


class ProcessStatistics(object):
    def __init__(self):
        self.release_rules = 0
        self.processed_releases = 0
        self.not_processed_releases = 0
        self.failed_releases = 0
        self.created_tickets = 0


class ReleaseProcessor(greenthread.GreenThread):

    name = 'release_processor'
    ITERATION_SLEEP = 5

    def __init__(self, release_selector, release_controller, metrics_registry, iteration_sleep=None):
        super(ReleaseProcessor, self).__init__()
        self.release_selector = release_selector
        self.release_controller = release_controller
        self._metrics_registry = metrics_registry
        self.log = logging.getLogger(__name__)
        self.iteration_sleep = iteration_sleep or self.ITERATION_SLEEP

    def process(self, release):
        return self.release_controller.process(release)

    def iterate(self):
        stats = ProcessStatistics()
        stats.release_rules, _ = self.release_controller.resync_storages_if_needed()
        stats.not_processed_releases, stats.failed_releases, batch = self.release_selector.select_next_batch()
        self.log.info('Selected %d not processed releases', stats.not_processed_releases)

        for r in batch:
            try:
                created_tickets = self.process(r)
                stats.created_tickets += created_tickets
                stats.processed_releases += 1
            except Exception:
                # We don't skip release requests (at least for now).
                # Let's just go on to the next.
                self.log.exception('Failed to process %s. Continue', r.meta.id)
        return stats

    def _collect_metrics(self, stats):
        self._metrics_registry.get_gauge('release_rules').set(stats.release_rules)
        self._metrics_registry.get_counter('processed_releases').inc(stats.processed_releases)
        self._metrics_registry.get_gauge('not_processed_releases').set(stats.not_processed_releases)
        self._metrics_registry.get_gauge('failed_releases').set(stats.failed_releases)
        self._metrics_registry.get_counter('created_tickets').inc(stats.created_tickets)

    def run(self):
        while True:
            try:
                timer = self._metrics_registry.get_histogram('iteration_time').timer()
                stats = self.iterate()
                timer.stop()
                self._collect_metrics(stats)
            except gevent.GreenletExit:
                raise
            except:
                self.log.exception('Iteration failed')
            gevent.sleep(self.iteration_sleep)
