from __future__ import unicode_literals

from infra.swatlib.gevent import geventutil as gutil
from infra.release_controller.src import consts
from infra.release_controller.src import releaseutil


def is_subdict(sub, d):
    for k, v in sub.iteritems():
        if d.get(k) != v:
            return False
    return True


def _match_sandbox_resource_types(release, release_rule):
    rule_resource_types = set(release_rule.spec.sandbox.resource_types)
    for patch in release_rule.spec.patches.itervalues():
        t = patch.sandbox.sandbox_resource_type
        if t:
            rule_resource_types.add(t)
    release_resource_types = [r.type for r in release.spec.sandbox.resources]
    return rule_resource_types.issubset(release_resource_types)


def _match_sandbox_resource_attributes(release, release_rule):
    rule_attributes = release_rule.spec.sandbox.attributes
    if not rule_attributes:
        return True
    for resource in release.spec.sandbox.resources:
        if is_subdict(rule_attributes, resource.attributes):
            return True
    return False


def _match_docker_images(release, release_rule):
    rule_image_names = set(i.name for i in release_rule.spec.docker.images)
    for patch in release_rule.spec.patches.itervalues():
        n = patch.docker.image_name
        if n:
            rule_image_names.add(n)
    release_image_names = set(i.name for i in release.spec.docker.images)
    return rule_image_names.issubset(release_image_names)


def _match_docker_images_legacy_rule(release, release_rule):
    release_image_names = set(i.name for i in release.spec.docker.images)
    return release_rule.spec.docker.image_name in release_image_names


class ReleaseMatcher(object):

    def __init__(self, release_rule_storage, stage_storage,
                 match_sandbox_resource_attributes):
        self.release_rule_storage = release_rule_storage
        self.stage_storage = stage_storage
        self._match_sandbox_resource_attributes = match_sandbox_resource_attributes

    def match(self, release):
        release_type = release.spec.WhichOneof('payload')
        if release_type == 'sandbox':
            return self._match_sandbox_release(release)
        elif release_type == 'docker':
            return self._match_docker_release(release)

    def _match_sandbox_release(self, release):
        result = []
        release_sb = release.spec.sandbox
        for rule in gutil.gevent_idle_iter(self.release_rule_storage.list_by_index(consts.SANDBOX_INDEX_NAME)):
            rule_sb = rule.spec.sandbox
            if rule_sb.task_type and release_sb.task_type != rule_sb.task_type:
                continue
            if rule_sb.release_types and release_sb.release_type not in rule_sb.release_types:
                continue
            if rule_sb.task_authors and release_sb.task_author not in rule_sb.task_authors:
                continue
            if not _match_sandbox_resource_types(release, rule):
                continue
            if self._match_sandbox_resource_attributes:
                if not _match_sandbox_resource_attributes(release, rule):
                    continue
            result.append(rule)
        return result

    def _match_docker_release(self, release):
        result = []
        for rule in gutil.gevent_idle_iter(self.release_rule_storage.list_by_index(consts.DOCKER_INDEX_NAME)):
            if rule.spec.docker.release_types and release.spec.docker.release_type not in rule.spec.docker.release_types:
                continue

            release_multi_docker = releaseutil.is_release_multi_docker(release)
            rule_multi_docker = releaseutil.is_rule_multi_docker(rule)
            if release_multi_docker and rule_multi_docker:
                # Release and rule support multi docker images. Use images
                # field to match.
                if not _match_docker_images(release, rule):
                    continue
            elif not release_multi_docker and not rule_multi_docker:
                # Neither release nor rule support multi docker images. User legacy
                # fields to match.
                if release.spec.docker.image_name != rule.spec.docker.image_name:
                    continue
            elif release_multi_docker and not rule_multi_docker:
                # Release supports multi docker images but rule does not.
                if not _match_docker_images_legacy_rule(release, rule):
                    continue
            elif not release_multi_docker and rule_multi_docker:
                # Release does not supports multi docker images but rule
                # supports. We ignore this case.
                continue
            result.append(rule)
        return result
