from __future__ import unicode_literals
import itertools

import yp.data_model

from infra.swatlib.gevent import geventutil as gutil


class NotProcessedReleaseBatchSelector(object):
    NOT_PROCESSED_QUERY = (
        '[/labels/deploy_engine] != "CI" AND '
        '([/status/processing] = null OR '
        '[/status/processing/finished/status] != "true")'
    )
    DEFAULT_BATCH_SIZE = 100
    META_PROCESSING_SELECTORS = ['/meta', '/status/processing/finished']

    def __init__(self, yp_client, batch_size=None):
        self.yp_client = yp_client
        if batch_size is None:
            batch_size = self.DEFAULT_BATCH_SIZE
        self.batch_size = batch_size
        self._ids = []
        self._current_id_index = 0
        self._failed_count = 0

    def _select_not_processed_release_ids(self):
        ts = self.yp_client.generate_timestamp()
        batches = self.yp_client.select_release_batches(
            query=self.NOT_PROCESSED_QUERY,
            selectors=self.META_PROCESSING_SELECTORS,
            batch_size=self.batch_size,
            timestamp=ts
        )
        releases = []
        failed_count = 0
        for r in gutil.gevent_idle_iter(itertools.chain.from_iterable(batches)):
            releases.append(r)
            if r.status.processing.finished.status == yp.data_model.CS_FALSE:
                failed_count += 1

        # TODO: implement idle_sort.
        releases = sorted(releases, key=lambda r: r.meta.creation_time)
        ids = []
        for r in gutil.gevent_idle_iter(releases):
            ids.append(r.meta.id)
        return ids, failed_count

    def _select_next_batch_ids(self):
        if self._current_id_index >= len(self._ids):
            self._ids, self._failed_count = self._select_not_processed_release_ids()
            self._current_id_index = 0
        rv = self._ids[self._current_id_index:self._current_id_index + self.batch_size]
        self._current_id_index += self.batch_size
        return rv

    def select_next_batch(self):
        ids = self._select_next_batch_ids()
        return len(self._ids), self._failed_count, self.yp_client.get_releases(ids)
