from __future__ import unicode_literals
import logging
import os

import flask
import gevent
import yp.client
from sepelib.core import config

from infra.swatlib import metrics
from infra.swatlib import webserver
from infra.ytexclusiveservice import yt_exclusiveservice

from infra.release_status_controller.src import controller
from infra.release_status_controller.src import indexers
from infra.release_status_controller.src import deploy_progress_maker
from infra.release_status_controller.src import patch_applied_cache
from infra.release_status_controller.src import patch_applied_checker
from infra.release_status_controller.src import patch_progress_maker
from infra.release_status_controller.src import processor
from infra.release_status_controller.src import release_status_updater
from infra.release_status_controller.src import ticket_status_maker
from infra.release_status_controller.src.lib import storage
from infra.release_status_controller.src.lib import yp_client


class Application(object):
    name = 'release_status_controller'

    def __init__(self, instance_id):
        self.instance_id = instance_id
        self.log = logging.getLogger(self.name)

    @staticmethod
    def setup_environment():
        # Patch requests connection pool to use gevent queue
        from requests.packages.urllib3.connectionpool import ConnectionPool
        from gevent.queue import LifoQueue

        ConnectionPool.QueueCls = LifoQueue
        # Disable requests spamming about:
        # Connection pool is full, discarding connection
        # This is the way we use alemate-http to avoid broken connections
        # There is nothing we can do about it, so simply mute
        logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.ERROR)
        logging.getLogger('ydb.connection').setLevel(logging.ERROR)

    @staticmethod
    def make_yp_client(config):
        token = config.get('token')
        if not token:
            config['token'] = os.environ.get('YP_TOKEN')
        base = yp.client.YpClient(address=config['address'], config=config)
        stub = base.create_grpc_object_stub()
        return yp_client.YpClient(stub=stub)

    def init_http_server(self, config, metrics_registry):
        app = flask.Flask('release_status_controller')
        app.add_url_rule('/ping', view_func=lambda: '')
        return webserver.WebServer(config.get(), app, version='0.0.1', metrics_registry=metrics_registry)

    def run(self):
        self.setup_environment()

        client = self.make_yp_client(config.get_value('yp'))

        # Make storages for release rules and stages.
        release_storage = storage.make_storage()
        ticket_storage = storage.make_storage(indexers.TICKET_INDEXERS)
        stage_storage = storage.make_storage()

        # Make patch progress maker.
        applied_checker = patch_applied_checker.PatchAppliedChecker()
        applied_cache = patch_applied_cache.PatchAppliedCache()
        progress_maker = patch_progress_maker.PatchProgressMaker(
            applied_checker=applied_checker,
            applied_cache=applied_cache
        )

        # Maker ticket status maker.
        status_maker = ticket_status_maker.TicketStatusMaker(
            patch_progress_maker=progress_maker,
            deploy_progress_maker=deploy_progress_maker.DeployProgressMaker(),
            stage_storage=stage_storage
        )

        # Maker release status updater.
        status_updater = release_status_updater.ReleaseStatusUpdater(
            yp_client=client,
            ticket_status_maker=status_maker,
            ticket_storage=ticket_storage
        )

        # Maker update status controller.
        status_ctl = controller.UpdateStatusController(
            yp_client=client,
            release_status_updater=status_updater,
            ticket_storage=ticket_storage,
            release_storage=release_storage,
            stage_storage=stage_storage
        )

        reg = metrics.Registry()
        # Make update status processor.
        proc = processor.UpdateStatusProcessor(status_controller=status_ctl,
                                               release_storage=release_storage,
                                               metrics_registry=reg)

        # self.processor = proc
        yt_manager_cfg = config.get_value('coord.manager')
        lock_name = config.get_value('coord.lock.name')
        expiration_timeout = config.get_value('coord.lock.expiration_timeout')
        acquire_timeout = config.get_value('coord.lock.acquire_timeout')
        self.processor = yt_exclusiveservice.YtExclusiveService(
            cfg=yt_manager_cfg,
            name=lock_name,
            runnable=proc,
            lock_timeout=expiration_timeout,
            acquire_timeout=acquire_timeout)

        self.log.info('Init and starting http server...')
        # Instantiate http server for stats.
        self.srv = self.init_http_server(config, metrics_registry=reg)
        # Run http server in separate thread.
        gevent.spawn(self.srv.run)

        self.log.info('Starting update status processor...')
        self.processor.start()
        self.processor.wait()

    def stop(self):
        self.log.info('Stopping update status processor...')
        self.processor.stop()
        self.log.info('Done.')
