from __future__ import unicode_literals
import logging

import gevent

from infra.swatlib.gevent import geventutil as gutil
from infra.release_status_controller.src.lib import storage


class UpdateStatusController(object):

    STAGE_SELECTORS = ['/labels', '/meta', '/spec', '/status']

    NOT_CLOSED_TICKET_QUERY = '[/status/progress/closed/status] != "true"'
    NOT_CLOSED_RELEASE_QUERY = '[/status/progress/closed/status] != "true" AND [/status/processing/finished/status] = "true"'
    CLOSED_TICKET_QUERY = '[/status/progress/closed/status] = "true"'

    def __init__(self, yp_client, release_status_updater,
                 ticket_storage, release_storage, stage_storage):
        self.yp_client = yp_client
        self.release_status_updater = release_status_updater
        self.ticket_storage = ticket_storage
        self.release_storage = release_storage
        self.stage_storage = stage_storage
        self.release_ids_with_closed_tickets = set()
        self.log = logging.getLogger(__name__)

    def select_not_closed_ticket_batches(self):
        return self.yp_client.select_deploy_ticket_batches(query=self.NOT_CLOSED_TICKET_QUERY)

    def select_not_closed_release_batches(self):
        return self.yp_client.select_release_batches(query=self.NOT_CLOSED_RELEASE_QUERY)

    def select_closed_ticket_release_id_batches(self):
        return self.yp_client.select_deploy_ticket_batches(query=self.CLOSED_TICKET_QUERY,
                                                           selectors=['/spec/release_id'],
                                                           batch_size=5000)

    # TODO: let's think about moving resync_* methods to StorageResyncer
    def _resync_with_batches(self, s, batches, key_getter):
        synced = storage.make_storage(s.indexers)
        for batch in batches:
            for obj in batch:
                synced.add(key=key_getter(obj), obj=obj)
            gevent.sleep(0)
        s.update_from_other_storage(synced)

    def resync_ticket_storage(self):
        batches = self.select_not_closed_ticket_batches()
        self._resync_with_batches(self.ticket_storage,
                                  batches,
                                  key_getter=storage.get_default_key_for_object)
        self.log.info('Ticket storage resynced. Count: %d', self.ticket_storage.count())

    def resync_release_storage(self):
        batches = self.select_not_closed_release_batches()
        self._resync_with_batches(self.release_storage,
                                  batches,
                                  key_getter=storage.get_default_key_for_object)
        self.log.info('Release storage resynced. Count: %d', self.release_storage.count())

    def resync_stage_storage(self):
        batches = self.yp_client.select_stage_batches(selectors=self.STAGE_SELECTORS,
                                                      fetch_timestamps=True)
        self._resync_with_batches(self.stage_storage,
                                  batches,
                                  key_getter=storage.get_default_key_for_object_with_timestamp)
        self.log.info('Stage storage resynced. Count: %d', self.stage_storage.count())

    def resync_storages(self):
        # We sync stages after releases and tickets to keep stage storage the
        # newest one.
        self.resync_release_storage()
        self.resync_ticket_storage()
        self.resync_stage_storage()

        self.log.info('Selecting release ids with closed tickets...')
        self.release_ids_with_closed_tickets.clear()
        for b in self.select_closed_ticket_release_id_batches():
            for t in gutil.gevent_idle_iter(b):
                self.release_ids_with_closed_tickets.add(t.spec.release_id)
        self.log.info('Release ids with closed tickets selected. Count: %d', len(self.release_ids_with_closed_tickets))

    def process(self, release):
        self.log.info('Updating release status: %s', release.meta.id)
        try:
            stats = self.release_status_updater.update_release_status(
                release=release,
                release_ids_with_closed_tickets=self.release_ids_with_closed_tickets
            )
        except Exception:
            self.log.exception('Release %s status updating failed', release.meta.id)
        else:
            self.log.info('Release %s status updated successfully', release.meta.id)
            return stats
