from __future__ import unicode_literals

from google.protobuf import message as pb_message
import yt_yson_bindings
import yt.yson as yson

from infra.swatlib.gevent import geventutil as gutil
from infra.release_status_controller.src.lib import pbutil


def get_deploy_unit_id_from_patch(patch):
    pl = patch.WhichOneof('payload')
    if pl == 'sandbox':
        return get_deploy_unit_id_from_sandbox_patch(patch)
    if pl == 'docker':
        return get_deploy_unit_id_from_docker_patch(patch)


def get_deploy_unit_id_from_sandbox_patch(patch):
    ref = patch.sandbox.WhichOneof('resource_ref')
    if ref == 'static':
        return patch.sandbox.static.deploy_unit_id
    raise ValueError('Cannot get deploy unit id from dynamic resource patch')


def get_deploy_unit_id_from_docker_patch(patch):
    return patch.docker.docker_image_ref.deploy_unit_id


def get_pod_template_spec_from_deploy_unit(deploy_unit):
    unit_type = deploy_unit.WhichOneof('pod_deploy_primitive')
    if unit_type == 'replica_set':
        return deploy_unit.replica_set.replica_set_template.pod_template_spec
    if unit_type == 'multi_cluster_replica_set':
        return deploy_unit.multi_cluster_replica_set.replica_set.pod_template_spec


def find_resource_in_sandbox_release(release, resource_type, attrs):
    for r in gutil.gevent_idle_iter(release.spec.sandbox.resources):
        if r.type == resource_type:
            attrs_matched = True
            for k, v in gutil.gevent_idle_iter(attrs.iteritems()):
                if r.attributes.get(k) != v:
                    attrs_matched = False
                    break
            if attrs_matched:
                return r


def progress_changed(old_p, new_p):
    return (pbutil.condition_changed(old_p.in_progress, new_p.in_progress) or
            pbutil.condition_changed(old_p.closed, new_p.closed))


def patch_progress_changed(old_p, new_p):
    return (pbutil.condition_changed(old_p.in_progress, new_p.in_progress) or
            pbutil.condition_changed(old_p.success, new_p.success) or
            pbutil.condition_changed(old_p.failed, new_p.failed) or
            pbutil.condition_changed(old_p.cancelled, new_p.cancelled))


def deploy_progress_changed(old_p, new_p):
    if set(old_p.cluster_statuses.iterkeys()) != set(new_p.cluster_statuses.iterkeys()):
        return True

    for k, old_cp in old_p.cluster_statuses.iteritems():
        new_cp = new_p.cluster_statuses[k]
        if (
            pbutil.condition_changed(old_cp.in_progress, new_cp.in_progress)
            or pbutil.condition_changed(old_cp.ready, new_cp.ready)
        ):
            return True

    return False


def is_progress_empty(p):
    return (pbutil.is_condition_unknown(p.in_progress) and
            pbutil.is_condition_unknown(p.closed))


def is_patch_progress_empty(p):
    return (pbutil.is_condition_unknown(p.in_progress) and
            pbutil.is_condition_unknown(p.success) and
            pbutil.is_condition_unknown(p.failed) and
            pbutil.is_condition_unknown(p.cancelled))


def is_patch_progress_closed(p):
    return (pbutil.is_condition_true(p.success) or
            pbutil.is_condition_true(p.failed) or
            pbutil.is_condition_true(p.cancelled))


class YsonObjectLoader(object):

    FULL_OBJECT_ATTR_PATH = ['']

    @staticmethod
    def make_attr_paths(selectors):
        return tuple(s.strip("/").split("/") for s in selectors)

    @classmethod
    def make_object_attr(cls, obj, attr_path):
        if not attr_path or attr_path == cls.FULL_OBJECT_ATTR_PATH:
            return None, None, obj
        parent = None
        attr = obj
        for name in attr_path:
            parent = attr
            attr = getattr(attr, name)
        return parent, attr_path[-1], attr

    def load_object_attrs(self, obj_class, paths, value_payloads):
        obj = obj_class()
        for i in xrange(len(paths)):
            parent, attr_name, attr = self.make_object_attr(obj, paths[i])
            yson_v = value_payloads[i].yson
            if isinstance(attr, pb_message.Message):
                o = yt_yson_bindings.loads_proto(yson_v,
                                                 type(attr),
                                                 skip_unknown_fields=True)
                attr.CopyFrom(o)
            else:
                setattr(parent, attr_name, yson.loads(yson_v))
        return obj

    def load_object(self, obj_class, selectors, value_payloads):
        paths = self.make_attr_paths(selectors)
        return self.load_object_attrs(obj_class, paths, value_payloads)


YSON_LOADER = YsonObjectLoader()
