from __future__ import unicode_literals

from infra.swatlib.gevent import geventutil as gutil
from infra.release_status_controller.src.lib import yputil


def make_sbr_url(resource_id):
    return 'sbr:{}'.format(resource_id)


def has_static_resource(pod_spec, resource_id, resource_url, sbr_url):
    resources = pod_spec.spec.pod_agent_payload.spec.resources.static_resources
    for r in gutil.gevent_idle_iter(resources):
        if r.id != resource_id:
            continue
        method = r.WhichOneof('download_method')
        if method != 'url':
            continue
        if r.url == resource_url or r.url == sbr_url:
            return True
    return False


def has_layer(pod_spec, layer_id, layer_url, sbr_url):
    layers = pod_spec.spec.pod_agent_payload.spec.resources.layers
    for layer in gutil.gevent_idle_iter(layers):
        if layer.id == layer_id and (layer.url == layer_url or layer.url == sbr_url):
            return True
    return False


def has_dyn_res_url(dyn_res, deploy_group_mark, url, sbr_url):
    for g in dyn_res.deploy_groups:
        if g.mark != deploy_group_mark:
            continue
        if url in g.urls or sbr_url in g.urls:
            return True
    return False


class DockerPatchAppliedChecker(object):

    @classmethod
    def is_patch_applied(cls, patch_spec, release, stage):
        unit_id = patch_spec.docker.docker_image_ref.deploy_unit_id
        box_id = patch_spec.docker.docker_image_ref.box_id
        unit_docker = stage.spec.deploy_units[unit_id].images_for_boxes[box_id]
        if release.spec.docker.images:
            release_image = None
            for image in release.spec.docker.images:
                if image.name == unit_docker.name:
                    release_image = image
                    break
            if not release_image:
                return False
            return (unit_docker.name == release_image.name and
                    unit_docker.tag == release_image.tag)
        return (unit_docker.name == release.spec.docker.image_name and
                unit_docker.tag == release.spec.docker.image_tag)


class SandboxPatchAppliedChecker(object):

    @classmethod
    def is_patch_applied(cls, patch_spec, release, stage):
        ref = patch_spec.sandbox.WhichOneof('resource_ref')
        if ref == 'static':
            return cls._is_static_resource_patch_applied(patch_spec,
                                                         release,
                                                         stage)
        if ref == 'dynamic':
            return cls._is_dynamic_resource_patch_applied(patch_spec,
                                                          release,
                                                          stage)

    @staticmethod
    def _is_static_resource_patch_applied(patch_spec, release, stage):
        unit_id = yputil.get_deploy_unit_id_from_sandbox_patch(patch_spec)
        unit = stage.spec.deploy_units[unit_id]
        pod_spec = yputil.get_pod_template_spec_from_deploy_unit(unit)
        resource = yputil.find_resource_in_sandbox_release(release,
                                                           patch_spec.sandbox.sandbox_resource_type,
                                                           patch_spec.sandbox.attributes)
        if resource is None:
            return False

        ref = patch_spec.sandbox.static.WhichOneof('resource_ref')
        if ref == 'static_resource_ref':
            return has_static_resource(pod_spec,
                                       patch_spec.sandbox.static.static_resource_ref,
                                       resource.skynet_id,
                                       make_sbr_url(resource.resource_id))
        if ref == 'layer_ref':
            return has_layer(pod_spec,
                             patch_spec.sandbox.static.layer_ref,
                             resource.skynet_id,
                             make_sbr_url(resource.resource_id))

    @staticmethod
    def _is_dynamic_resource_patch_applied(patch_spec, release, stage):
        dyn_res = stage.spec.dynamic_resources[patch_spec.sandbox.dynamic.dynamic_resource_id]
        resource = yputil.find_resource_in_sandbox_release(release,
                                                           patch_spec.sandbox.sandbox_resource_type,
                                                           patch_spec.sandbox.attributes)
        if resource is None:
            return False
        return has_dyn_res_url(dyn_res=dyn_res.dynamic_resource,
                               deploy_group_mark=patch_spec.sandbox.dynamic.deploy_group_mark,
                               url=resource.skynet_id,
                               sbr_url=make_sbr_url(resource.resource_id))


class PatchAppliedChecker(object):

    @classmethod
    def is_patch_applied(cls, patch_spec, release, stage):
        patch_type = patch_spec.WhichOneof('payload')
        if patch_type == 'sandbox':
            return SandboxPatchAppliedChecker.is_patch_applied(patch_spec,
                                                               release,
                                                               stage)
        if patch_type == 'docker':
            return DockerPatchAppliedChecker.is_patch_applied(patch_spec,
                                                              release,
                                                              stage)
