from __future__ import unicode_literals

import yp.data_model

from infra.release_status_controller.src.lib import pbutil
from infra.release_status_controller.src.lib import progressutil
from infra.release_status_controller.src.lib import yputil


class DeployUnitStatusAdapter(object):

    def __init__(self, deploy_unit_status):
        self.deploy_unit_status = deploy_unit_status

    def ready(self):
        return self.deploy_unit_status.ready

    def in_progress(self):
        return self.deploy_unit_status.in_progress

    def failed(self):
        return self.deploy_unit_status.failed


class DynResourceStatusAdapter(object):

    def __init__(self, dyn_resource_status):
        self.dyn_resource_status = dyn_resource_status

    def ready(self):
        return self.dyn_resource_status.ready.condition

    def in_progress(self):
        return self.dyn_resource_status.in_progress.condition

    def failed(self):
        return self.dyn_resource_status.error.condition


class PatchProgressMaker(object):

    def __init__(self, applied_checker, applied_cache):
        self.applied_checker = applied_checker
        self.applied_cache = applied_cache

    def _is_patch_applied(self, patch_name, patch_spec, release, stage, stage_spec_ts):
        is_applied, exists = self.applied_cache.get(patch_name,
                                                    release.meta.id,
                                                    stage.meta.id,
                                                    stage_spec_ts)
        if exists:
            return is_applied
        is_applied = self.applied_checker.is_patch_applied(patch_spec=patch_spec,
                                                           release=release,
                                                           stage=stage)
        if is_applied is not None:
            self.applied_cache.set(patch_name,
                                   release.meta.id,
                                   stage.meta.id,
                                   stage_spec_ts,
                                   is_applied)
        return is_applied

    @staticmethod
    def _fill_deploy_unit_patch_progress_conditions(progress, status_adapter, start_time):
        if pbutil.is_condition_true(status_adapter.failed()):
            progressutil.set_failed_from_condition(progress,
                                                   status_adapter.failed(),
                                                   start_time=start_time)
            return
        if pbutil.is_condition_true(status_adapter.in_progress()):
            progressutil.set_in_progress_from_condition(progress,
                                                        status_adapter.in_progress(),
                                                        start_time=start_time)
            return
        if pbutil.is_condition_true(status_adapter.ready()):
            progressutil.set_success_from_condition(progress,
                                                    status_adapter.ready(),
                                                    start_time=start_time)

    @staticmethod
    def _fill_dyn_resource_patch_progress_conditions(progress, status_adapter, start_time):
        if pbutil.is_condition_true(status_adapter.ready()):
            progressutil.set_success_from_condition(progress,
                                                    status_adapter.ready(),
                                                    start_time=start_time)
            return
        if pbutil.is_condition_true(status_adapter.failed()):
            progressutil.set_failed_from_condition(progress,
                                                   status_adapter.failed(),
                                                   start_time=start_time)
        if pbutil.is_condition_true(status_adapter.in_progress()):
            progressutil.set_in_progress_from_condition(progress,
                                                        status_adapter.in_progress(),
                                                        start_time=start_time)

    def make_patch_progress(self, patch_name, patch_spec, patch_status, release, stage, stage_spec_ts):
        progress = yp.data_model.TDeployPatchProgress()
        start_time = patch_status.progress.start_time

        if patch_status.action.type == yp.data_model.DPAT_SKIP:
            progressutil.set_cancelled(progress,
                                       message='Patch is skipped',
                                       start_time=start_time)
            return progress

        if stage.status.spec_timestamp != stage_spec_ts:
            return progress

        if patch_status.action.type != yp.data_model.DPAT_COMMIT:
            return progress

        if not self._is_patch_applied(patch_name=patch_name,
                                      patch_spec=patch_spec,
                                      release=release,
                                      stage=stage,
                                      stage_spec_ts=stage_spec_ts):
            progressutil.set_cancelled(progress,
                                       message='Patch deploy is cancelled',
                                       start_time=start_time)
        else:
            payload = patch_spec.WhichOneof('payload')
            if payload == 'docker':
                unit_id = yputil.get_deploy_unit_id_from_docker_patch(patch_spec)
                unit_status = stage.status.deploy_units[unit_id]
                status_adapter = DeployUnitStatusAdapter(unit_status)
                self._fill_deploy_unit_patch_progress_conditions(progress,
                                                                 status_adapter,
                                                                 start_time)
            elif payload == 'sandbox':
                ref = patch_spec.sandbox.WhichOneof('resource_ref')
                if ref == 'static':
                    unit_id = yputil.get_deploy_unit_id_from_sandbox_patch(patch_spec)
                    unit_status = stage.status.deploy_units[unit_id]
                    status_adapter = DeployUnitStatusAdapter(unit_status)
                    self._fill_deploy_unit_patch_progress_conditions(progress,
                                                                     status_adapter,
                                                                     start_time)
                if ref == 'dynamic':
                    dyn_resource_status = stage.status.dynamic_resources[patch_spec.sandbox.dynamic.dynamic_resource_id]
                    status_adapter = DynResourceStatusAdapter(dyn_resource_status.status)
                    self._fill_dyn_resource_patch_progress_conditions(progress,
                                                                      status_adapter,
                                                                      start_time)
        return progress
