from __future__ import unicode_literals

import logging

import gevent

from infra.swatlib.gevent import geventutil as gutil
from infra.swatlib.gevent import greenthread


class ProcessStatistics(object):
    def __init__(self):
        self.pending_releases = 0
        self.in_progress_releases = 0
        self.closed_releases = 0
        self.pending_tickets = 0
        self.in_progress_tickets = 0
        self.closed_tickets = 0

    def update_by_release_statistics(self, release_stats):
        if release_stats.pending:
            self.pending_releases += 1
        elif release_stats.in_progress:
            self.in_progress_releases += 1
        elif release_stats.closed:
            self.closed_releases += 1
        self.pending_tickets += release_stats.pending_tickets
        self.in_progress_tickets += release_stats.in_progress_tickets
        self.closed_tickets += release_stats.closed_tickets


class UpdateStatusProcessor(greenthread.GreenThread):

    name = 'update_status_processor'
    ITERATION_SLEEP = 5

    def __init__(self, status_controller, release_storage, metrics_registry,
                 iteration_sleep=None):
        super(UpdateStatusProcessor, self).__init__()
        self.status_controller = status_controller
        self.release_storage = release_storage
        self._metrics_registry = metrics_registry
        self.log = logging.getLogger(__name__)
        self.iteration_sleep = iteration_sleep or self.ITERATION_SLEEP

    def process(self, release):
        return self.status_controller.process(release)

    def iterate(self):
        stats = ProcessStatistics()
        self.log.info('Resyncing storages')
        self.status_controller.resync_storages()
        self.log.info('Storages resynced')
        for r in gutil.gevent_idle_iter(self.release_storage.list()):
            try:
                release_stats = self.process(r)
            except Exception:
                # We don't skip release requests (at least for now).
                # Let's just go on to the next.
                self.log.exception('Failed to update release status: %s. Continue', r.meta.id)
            else:
                if release_stats:
                    stats.update_by_release_statistics(release_stats)
        return stats

    def _collect_metrics(self, stats):
        self._metrics_registry.get_gauge('pending_releases').set(stats.pending_releases)
        self._metrics_registry.get_gauge('in_progress_releases').set(stats.in_progress_releases)
        self._metrics_registry.get_counter('closed_releases').inc(stats.closed_releases)
        self._metrics_registry.get_gauge('pending_tickets').set(stats.pending_tickets)
        self._metrics_registry.get_gauge('in_progress_tickets').set(stats.in_progress_tickets)
        self._metrics_registry.get_counter('closed_tickets').inc(stats.closed_tickets)

    def run(self):
        while True:
            try:
                timer = self._metrics_registry.get_histogram('iteration_time').timer()
                stats = self.iterate()
                timer.stop()
                self._collect_metrics(stats)
            except gevent.GreenletExit:
                raise
            except:
                self.log.exception('Iteration failed')
            gevent.sleep(self.iteration_sleep)
