from __future__ import unicode_literals
import logging

import yp.data_model

from infra.swatlib.gevent import geventutil as gutil
from infra.release_status_controller.src import aggregated_progress
from infra.release_status_controller.src import indexers
from infra.release_status_controller.src.lib import pbutil
from infra.release_status_controller.src.lib import yputil


class ReleaseStatistics(object):

    def __init__(self):
        self.pending = False
        self.in_progress = False
        self.closed = False
        self.pending_tickets = 0
        self.in_progress_tickets = 0
        self.closed_tickets = 0

    def update_by_ticket_progress(self, progress):
        if yputil.is_progress_empty(progress):
            self.pending_tickets += 1
        elif pbutil.is_condition_true(progress.in_progress):
            self.in_progress_tickets += 1
        elif pbutil.is_condition_true(progress.closed):
            self.closed_tickets += 1

    def update_by_release_aggregated_progress(self, progress):
        if progress.is_empty:
            self.pending = True
        elif progress.in_progress:
            self.in_progress = True
        elif progress.closed:
            self.closed = True


class ReleaseStatusUpdater(object):

    CLOSED_QUERY_TPL = '[/spec/release_id] = "{}" AND [/status/progress/closed/status] = "true"'

    def __init__(self, yp_client, ticket_status_maker, ticket_storage):
        self.yp_client = yp_client
        self.ticket_status_maker = ticket_status_maker
        self.ticket_storage = ticket_storage
        self.log = logging.getLogger(__name__)

    def update_release_status(self, release, release_ids_with_closed_tickets):
        stats = ReleaseStatistics()
        status = yp.data_model.TReleaseStatus()
        agg = aggregated_progress.AggregatedProgress()
        release_id = release.meta.id
        updated_tickets = 0
        for t in gutil.gevent_idle_iter(self.ticket_storage.find_in_index(indexers.RELEASE_ID_INDEX, release_id)):
            s, changed = self.ticket_status_maker.make_ticket_status(t, release)
            if changed:
                self.yp_client.update_deploy_ticket_status_progress(deploy_ticket_id=t.meta.id,
                                                                    status=s)
                updated_tickets += 1
                self.log.info('Updated ticket status: %s', t.meta.id)
            else:
                self.log.info('Ticket status is not changed: %s', t.meta.id)
            if not agg.in_progress:
                agg.update_by_ticket_progress(s.progress)
            stats.update_by_ticket_progress(s.progress)

        if agg.is_empty:
            # Aggregated status is empty. This means that all not closed tickets
            # are still not committed. We need to check here if release has
            # closed tickets.
            # TODO: fetch releases and tickets with timestamps.
            if release.meta.id in release_ids_with_closed_tickets:
                agg.update(is_empty=False, in_progress=False, closed=True)

        status.progress.CopyFrom(agg.make_release_progress(status=release.status))
        changed = yputil.progress_changed(release.status.progress,
                                          status.progress)
        self.log.info('Updated %d tickets for release %s', updated_tickets, release_id)
        if changed:
            self.yp_client.update_release_status_progress(release_id=release_id,
                                                          progress=status.progress)
            self.log.info('Updated release status: %s', release_id)
        else:
            self.log.info('Release status is not changed: %s', release_id)
        stats.update_by_release_aggregated_progress(agg)
        return stats
