#include "resource_cache_gc_manager.h"

#include <infra/libs/sensors/macros.h>

namespace NInfra::NResourceCacheController {

namespace {

const TSensorGroup RESOURCE_CACHE_GC_MANAGER_SENSOR_GROUP("resource_cache_gc_manager");

} // namespace

TString TResourceCacheGCManager::GetObjectId() const {
    return PodId_;
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument> TResourceCacheGCManager::GetDependentObjectsSelectArguments() const {
    return {{
        NYP::NClient::NApi::NProto::OT_RESOURCE_CACHE
        , {
            "/meta/id"
        } /* selector */
        , TStringBuilder() << "[/meta/pod_set_id] = '" << PodSetId_ << "'" /* filter */
        , {} /* options */
        , {} /* clientFilterSelectors */
        , {} /* overrideYpReqLimitsConfig */
        , true /* selectAll */
        , Nothing() /*clusterName */
    }};
}

void TResourceCacheGCManager::GenerateYpUpdates(
    const ISingleClusterObjectManager::TDependentObjects& dependentObjects
    , TVector<ISingleClusterObjectManager::TRequest>& requests
    , TLogFramePtr /* frame */
) const {
    // No resource cache is attached to this pod
    if (dependentObjects.SelectedObjects[0]->Results.empty()) {
        STATIC_INFRA_RATE_SENSOR(RESOURCE_CACHE_GC_MANAGER_SENSOR_GROUP, "clear_pod_resource_cache_spec");

        requests.emplace_back(
            NYP::NClient::TUpdateRequest(
                NYP::NClient::NApi::NProto::OT_POD
                , PodId_
                , {} /* SetVec_ */
                , {
                    NYP::NClient::TRemoveRequest("/spec/resource_cache/spec")
                } /* RemoveVec_ */
            )
        );
    } else {
        STATIC_INFRA_RATE_SENSOR(RESOURCE_CACHE_GC_MANAGER_SENSOR_GROUP, "skip_pod_with_resource_cache_spec");
    }
}

TResourceCacheGCManagerFactory::TResourceCacheGCManagerFactory(
    const TResourceCacheGCManagerFactoryConfig& config
    , NController::TShardPtr shard
)
    : ISingleClusterObjectManagerFactory(
        "resource_cache_gc_manager_factory"
        , shard
        , true
        , FromString<TDuration>(config.GetSyncInterval())
    )
{
}

NController::ISingleClusterObjectManager::TSelectArgument TResourceCacheGCManagerFactory::GetSelectArgument(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */, NInfra::TLogFramePtr) const {
    return {
        NYP::NClient::NApi::NProto::OT_POD
        , {
            "/meta/id"
            , "/meta/pod_set_id"
        } /* selector */
        , "[/spec/resource_cache/spec] != #" /* filter */
        , {} /* options */
        , {} /* clientFilterSelectors */
        , {} /* overrideYpReqLimitsConfig */
        , true /* selectAll */
        , Nothing() /*clusterName */
    };
}

TExpected<NController::TSingleClusterObjectManagerPtr, TResourceCacheGCManagerFactory::TValidationError> TResourceCacheGCManagerFactory::GetSingleClusterObjectManager(
    const NController::TSelectorResultPtr& selectorResultPtr
    , TLogFramePtr /* frame */
) const {
    TString podId;
    TString podSetId;

    selectorResultPtr->Fill(
        &podId
        , &podSetId
    );

    return NController::TSingleClusterObjectManagerPtr(
        new TResourceCacheGCManager(
            podId
            , podSetId
        )
    );
}

} // namespace NInfra::NResourceCacheController
