#include "resource_cache_gc_manager.h"

#include "test_functions.h"

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NResourceCacheController::NResourceCacheGCManagerTest {

static TLogger logger({});

const TString RESOURCE_CACHE_ID = "resource_cache_id";
const TString POD_SET_ID = "pod_set_id";
const TString POD_ID = "pod_id";

const TString SYNC_INTERVAL_STR = "2m";
const TDuration SYNC_INTERVAL = TDuration::Minutes(2);

NYP::NClient::NApi::NProto::TAttributeList GetResourceCacheAtributeList(
    const TString& resourceCacheId = RESOURCE_CACHE_ID
) {
    NYP::NClient::NApi::NProto::TAttributeList attributeList;
    attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(resourceCacheId));

    return attributeList;
}

NYP::NClient::NApi::NProto::TAttributeList GetPodAtributeList(
    const TString& podId = POD_ID
    , const TString& podSetId = POD_SET_ID
) {
    NYP::NClient::NApi::NProto::TAttributeList attributeList;
    attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(podId));
    attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(podSetId));

    return attributeList;
}

TResourceCacheGCManagerFactoryConfig GetResourceCacheGCManagerFactoryConfig(
    const TString syncInterval = SYNC_INTERVAL_STR
) {
    TResourceCacheGCManagerFactoryConfig config;
    config.SetSyncInterval(syncInterval);
    return config;
}

Y_UNIT_TEST_SUITE(ResourceCacheGCManagerTestSuite) {

Y_UNIT_TEST(TestFactorySelectArgument) {
    NController::TSharding shardFactory;
    TResourceCacheGCManagerFactory factory(GetResourceCacheGCManagerFactoryConfig(), shardFactory.GetShard(0));

    auto select = factory.GetSelectArgument();
    UNIT_ASSERT_EQUAL(NYP::NClient::NApi::NProto::OT_POD, select.ObjectType);
    UNIT_ASSERT_EQUAL_C(2, select.Selector.size(), select.Selector.size());
    UNIT_ASSERT_STRINGS_EQUAL("/meta/id", select.Selector[0]);
    UNIT_ASSERT_STRINGS_EQUAL("/meta/pod_set_id", select.Selector[1]);

    UNIT_ASSERT_STRINGS_EQUAL(select.Filter, "[/spec/resource_cache/spec] != #");
}

Y_UNIT_TEST(TestFactoryCustomSyncInterval) {
    NController::TSharding shardFactory;
    TResourceCacheGCManagerFactory factory(GetResourceCacheGCManagerFactoryConfig(), shardFactory.GetShard(0));

    UNIT_ASSERT(factory.GetCustomSyncInterval().Defined());
    UNIT_ASSERT_EQUAL(*factory.GetCustomSyncInterval(), SYNC_INTERVAL);
}

Y_UNIT_TEST(TestManagerDependentObjects) {
    NController::TSharding shardFactory;
    TResourceCacheGCManagerFactory factory(GetResourceCacheGCManagerFactoryConfig(), shardFactory.GetShard(0));

    auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetPodAtributeList());

    auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
    UNIT_ASSERT(result);
    auto manager = result.Success();

    auto dependentObjects = manager->GetDependentObjectsSelectArguments()[0];
    UNIT_ASSERT_EQUAL(NYP::NClient::NApi::NProto::OT_RESOURCE_CACHE, dependentObjects.ObjectType);

    UNIT_ASSERT_EQUAL_C(1, dependentObjects.Selector.size(), dependentObjects.Selector.size());
    UNIT_ASSERT_STRINGS_EQUAL("/meta/id", dependentObjects.Selector[0]);

    UNIT_ASSERT_STRINGS_EQUAL(dependentObjects.Filter, "[/meta/pod_set_id] = 'pod_set_id'");
}

Y_UNIT_TEST(TestPodWithResourceCache) {
    NController::TSharding shardFactory;
    TResourceCacheGCManagerFactory factory(GetResourceCacheGCManagerFactoryConfig(), shardFactory.GetShard(0));

    auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetPodAtributeList());

    auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
    UNIT_ASSERT(result);
    auto manager = result.Success();

    NController::ISingleClusterObjectManager::TDependentObjects dependentObjectsResult = {
        {
            MakeAtomicShared<NController::TSelectObjectsResult>(
                TVector<NController::TSelectorResultPtr>{MakeAtomicShared<NYP::NClient::TSelectorResult>(GetResourceCacheAtributeList())}
            )
        }  /* SelectedObjects */
        , {}  /* ObjectsAccessAllowedUserIds_ */
        , {}  /* GotObjects */
    };

    TVector<NYP::NClient::TCreateObjectRequest> create;
    TVector<NYP::NClient::TRemoveObjectRequest> remove;
    TVector<NYP::NClient::TUpdateRequest> update;
    NManagerFactoryTestLib::GenerateYpUpdates(manager, dependentObjectsResult, create, remove, update);

    UNIT_ASSERT(create.empty());
    UNIT_ASSERT(remove.empty());
    UNIT_ASSERT(update.empty());
}

Y_UNIT_TEST(TestPodWithoutResourceCache) {
    NController::TSharding shardFactory;
    TResourceCacheGCManagerFactory factory(GetResourceCacheGCManagerFactoryConfig(), shardFactory.GetShard(0));

    auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetPodAtributeList());

    auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
    UNIT_ASSERT(result);
    auto manager = result.Success();

    NController::ISingleClusterObjectManager::TDependentObjects dependentObjectsResult = {
        {
            MakeAtomicShared<NController::TSelectObjectsResult>()
        }  /* SelectedObjects */
        , {}  /* ObjectsAccessAllowedUserIds_ */
        , {}  /* GotObjects */
    };

    TVector<NYP::NClient::TCreateObjectRequest> create;
    TVector<NYP::NClient::TRemoveObjectRequest> remove;
    TVector<NYP::NClient::TUpdateRequest> update;
    NManagerFactoryTestLib::GenerateYpUpdates(manager, dependentObjectsResult, create, remove, update);

    UNIT_ASSERT(create.empty());
    UNIT_ASSERT(remove.empty());

    UNIT_ASSERT_EQUAL_C(update.size(), 1, update.size());

    UNIT_ASSERT_EQUAL(update[0].GetObjectType(), NYP::NClient::NApi::NProto::OT_POD);
    UNIT_ASSERT_STRINGS_EQUAL(update[0].GetObjectId(), POD_ID);

    UNIT_ASSERT_EQUAL_C(update[0].GetRemoveVec().size(), 1, update[0].GetRemoveVec().size());
    UNIT_ASSERT_STRINGS_EQUAL(update[0].GetRemoveVec()[0].GetPath(), "/spec/resource_cache/spec");
}

}

} // namespace NInfra::NResourceCacheController::NResourceCacheGCManagerTest
