#include "resource_cache_manager.h"

#include <infra/resource_cache_controller/libs/resource_cache_controller/resource_cache_controller.h>

#include <infra/libs/sensors/macros.h>

namespace NInfra::NResourceCacheController {

namespace {

const TSensorGroup RESOURCE_CACHE_MANAGER_SENSOR_GROUP("resource_cache_manager");

} // namespace

TString TResourceCacheManager::GetObjectId() const {
    return ResourceCache_.Meta().id();
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument> TResourceCacheManager::GetDependentObjectsSelectArguments() const {
    return {{
        NYP::NClient::NApi::NProto::OT_POD
        , {
            "/meta/id"
            , "/spec/resource_cache"
            , "/status/agent/pod_agent_payload/status/resource_cache"
        } /* selector */
        , TStringBuilder() << "[/meta/pod_set_id] = '" << ResourceCache_.Meta().pod_set_id() << "'" /* filter */
        , {} /* options */
        , {} /* clientFilterSelectors */
        , {} /* overrideYpReqLimitsConfig */
        , true /* selectAll */
        , Nothing() /* clusterName */
    }};
}

void TResourceCacheManager::GenerateYpUpdates(
    const ISingleClusterObjectManager::TDependentObjects& dependentObjects
    , TVector<ISingleClusterObjectManager::TRequest>& requests
    , TLogFramePtr /* frame */
) const {
    TResourceCacheController controller(ResourceCache_);

    auto pods = FillPods(dependentObjects.SelectedObjects[0]->Results);
    STATIC_INFRA_RATE_SENSOR_X(RESOURCE_CACHE_MANAGER_SENSOR_GROUP, "selected_pods", pods.size());

    controller.ApplySpecToCurrentStatus();
    controller.AddPodsStatusToCurrentStatus(pods);

    for (auto&& update : controller.GenerateUpdates(pods)) {
        requests.emplace_back(std::move(update));
    }
}

TVector<NYP::NClient::TPod> TResourceCacheManager::FillPods(
    const TVector<NController::TSelectorResultPtr>& selectResults
) const {
    TVector<NYP::NClient::TPod> pods(selectResults.size());
    for (size_t i = 0; i < pods.size(); ++i) {
        selectResults[i]->Fill(
            pods[i].MutableMeta()->mutable_id()
            , pods[i].MutableSpec()->mutable_resource_cache()
            , pods[i].MutableStatus()->mutable_agent()->mutable_pod_agent_payload()->mutable_status()->mutable_resource_cache()
        );
    }

    return pods;
}

TResourceCacheManagerFactory::TResourceCacheManagerFactory(
    NController::TShardPtr shard
)
    : ISingleClusterObjectManagerFactory("resource_cache_manager_factory", shard)
{
}

NController::ISingleClusterObjectManager::TSelectArgument TResourceCacheManagerFactory::GetSelectArgument(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */, NInfra::TLogFramePtr) const {
    return {
        NYP::NClient::NApi::NProto::OT_RESOURCE_CACHE
        , {
            "/meta/id"
            , "/meta/pod_set_id"
            , "/spec"
            , "/status"
        } /* selector */
        , "" /* filter */
        , {} /* options */
        , {} /* clientFilterSelectors */
        , {} /* overrideYpReqLimitsConfig */
        , true /* selectAll */
        , Nothing() /* clusterName */
    };
}

TExpected<NController::TSingleClusterObjectManagerPtr, TResourceCacheManagerFactory::TValidationError> TResourceCacheManagerFactory::GetSingleClusterObjectManager(
    const NController::TSelectorResultPtr& selectorResultPtr
    , TLogFramePtr /* frame */
) const {
    NYP::NClient::TResourceCache resourceCache;

    selectorResultPtr->Fill(
        resourceCache.MutableMeta()->mutable_id()
        , resourceCache.MutableMeta()->mutable_pod_set_id()
        , resourceCache.MutableSpec()
        , resourceCache.MutableStatus()
    );

    // ClearCachedResourceIdInSpecs
    for (auto& cachedResource : *resourceCache.MutableSpec()->mutable_cached_resources()) {
        if (cachedResource.has_layer() && !cachedResource.layer().id().empty()) {
            cachedResource.mutable_layer()->clear_id();
        }
        if (cachedResource.has_static_resource() && !cachedResource.static_resource().id().empty()) {
            cachedResource.mutable_static_resource()->clear_id();
        }
    }

    if (resourceCache.Meta().pod_set_id().empty()) {
        return TValidationError{resourceCache.ObjectType, resourceCache.Meta().id(), "Pod set id is empty"};
    }

    THashSet<TString> cachedResourceIds;
    for (const auto& cachedResource : resourceCache.Spec().cached_resources()) {
        if (cachedResource.has_layer() && cachedResource.has_static_resource()) {
            return TValidationError{resourceCache.ObjectType, resourceCache.Meta().id(), "One of cached resources has both layer ans static_resource spec"};
        }
        if (cachedResourceIds.contains(cachedResource.id())) {
            return TValidationError{
                resourceCache.ObjectType
                , resourceCache.Meta().id()
                , TStringBuilder() << "Two or more cached_resources has same id: '" << cachedResource.id() << "'"
            };
        }
        cachedResourceIds.insert(cachedResource.id());
    }

    return NController::TSingleClusterObjectManagerPtr(new TResourceCacheManager(resourceCache));
}

} // namespace NInfra::NResourceCacheController
