#include "resource_cache_manager.h"

#include "test_functions.h"

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NResourceCacheController::NResourceCacheManagerTest {

static TLogger logger({});

const TString RESOURCE_CACHE_ID = "resource_cache_id";
const TString POD_SET_ID = "pod_set_id";
const TString POD_ID = "pod_id";

NYP::NClient::NApi::NProto::TAttributeList GetResourceCacheAtributeList(
    const TString& resourceCacheId = RESOURCE_CACHE_ID
    , const TString& podSetId = POD_SET_ID
    , ui32 revision = 1
    , TVector<NYP::NClient::NApi::NProto::TResourceCacheSpec::TCachedResource> cacheResources= {}
) {
    NYP::NClient::NApi::NProto::TAttributeList attributeList;
    attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(resourceCacheId));
    attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(podSetId));
    {
        NYP::NClient::NApi::NProto::TResourceCacheSpec resourceCacheSpec;
        resourceCacheSpec.set_revision(revision);

        for (const auto& cacheResource : cacheResources) {
            resourceCacheSpec.add_cached_resources()->CopyFrom(cacheResource);
        }

        attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(resourceCacheSpec));
    }
    {
        NYP::NClient::NApi::NProto::TResourceCacheStatus resourceCacheStatus;
        resourceCacheStatus.set_revision(revision);

        attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(resourceCacheStatus));
    }

    return attributeList;
}

NYP::NClient::NApi::NProto::TResourceCacheSpec::TCachedResource GetSimpleCachedLayer(
    const TString& id
    , ui32 maxLatestRevisions
    , const TString& url
    , const TString& checksum
) {
    NYP::NClient::NApi::NProto::TResourceCacheSpec::TCachedResource cachedLayer;
    cachedLayer.set_id(id);
    cachedLayer.mutable_basic_strategy()->set_max_latest_revisions(maxLatestRevisions);
    cachedLayer.mutable_layer()->set_url(url);
    cachedLayer.mutable_layer()->set_checksum(checksum);

    return cachedLayer;
}

NYP::NClient::NApi::NProto::TResourceCacheSpec::TCachedResource GetSimpleCachedStaticResource(
    const TString& id
    , ui32 maxLatestRevisions
    , const TString& url
    , const TString& checksum
    , ui32 checkPeriodMs
) {
    NYP::NClient::NApi::NProto::TResourceCacheSpec::TCachedResource cachedStaticResource;
    cachedStaticResource.set_id(id);
    cachedStaticResource.mutable_basic_strategy()->set_max_latest_revisions(maxLatestRevisions);
    cachedStaticResource.mutable_static_resource()->set_url(url);
    cachedStaticResource.mutable_static_resource()->mutable_verification()->set_checksum(checksum);
    cachedStaticResource.mutable_static_resource()->mutable_verification()->set_check_period_ms(checkPeriodMs);

    return cachedStaticResource;
}

NYP::NClient::NApi::NProto::TAttributeList GetPodAtributeList(
    const TString& podId = POD_ID
) {
    NYP::NClient::NApi::NProto::TAttributeList attributeList;
    attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(podId));

    {
        NYP::NClient::NApi::NProto::TPodSpec::TPodAgentResourceCache podResourceCacheSpec;

        attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(podResourceCacheSpec));
    }
    {
        NInfra::NPodAgent::API::TResourceGangStatus podResourceCacheStatus;

        attributeList.add_value_payloads()->set_yson(NManagerFactoryTestLib::ToYson(podResourceCacheStatus));
    }

    return attributeList;
}

// All main tests in libs/resource_cache_controller/tests
// TODO(chegoryu) move them here
Y_UNIT_TEST_SUITE(ResourceCacheManagerTestSuite) {

Y_UNIT_TEST(TestFactorySelectArgument) {
    NController::TSharding shardFactory;
    TResourceCacheManagerFactory factory(shardFactory.GetShard(0));

    auto select = factory.GetSelectArgument();
    UNIT_ASSERT_EQUAL(NYP::NClient::NApi::NProto::OT_RESOURCE_CACHE, select.ObjectType);
    UNIT_ASSERT_EQUAL_C(4, select.Selector.size(), select.Selector.size());
    UNIT_ASSERT_STRINGS_EQUAL("/meta/id", select.Selector[0]);
    UNIT_ASSERT_STRINGS_EQUAL("/meta/pod_set_id", select.Selector[1]);
    UNIT_ASSERT_STRINGS_EQUAL("/spec", select.Selector[2]);
    UNIT_ASSERT_STRINGS_EQUAL("/status", select.Selector[3]);

    UNIT_ASSERT_STRINGS_EQUAL(select.Filter, "");
}

Y_UNIT_TEST(TestManagerDependentObjects) {
    NController::TSharding shardFactory;
    TResourceCacheManagerFactory factory(shardFactory.GetShard(0));

    auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(GetResourceCacheAtributeList());

    auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
    UNIT_ASSERT(result);
    auto manager = result.Success();

    auto dependentObjects = manager->GetDependentObjectsSelectArguments()[0];
    UNIT_ASSERT_EQUAL(NYP::NClient::NApi::NProto::OT_POD, dependentObjects.ObjectType);

    UNIT_ASSERT_EQUAL_C(3, dependentObjects.Selector.size(), dependentObjects.Selector.size());
    UNIT_ASSERT_STRINGS_EQUAL("/meta/id", dependentObjects.Selector[0]);
    UNIT_ASSERT_STRINGS_EQUAL("/spec/resource_cache", dependentObjects.Selector[1]);
    UNIT_ASSERT_STRINGS_EQUAL("/status/agent/pod_agent_payload/status/resource_cache", dependentObjects.Selector[2]);

    UNIT_ASSERT_STRINGS_EQUAL(dependentObjects.Filter, "[/meta/pod_set_id] = 'pod_set_id'");
}


Y_UNIT_TEST(TestSimpleUpdate) {
    NController::TSharding shardFactory;
    TResourceCacheManagerFactory factory(shardFactory.GetShard(0));

    auto selectorResult = MakeAtomicShared<NYP::NClient::TSelectorResult>(
        GetResourceCacheAtributeList(
            RESOURCE_CACHE_ID
            , POD_SET_ID
            , 1
            , {
                GetSimpleCachedLayer(
                    "my_layer"
                    , 5
                    , "layer_url"
                    , "layer_checksum"
                )
                , GetSimpleCachedStaticResource(
                    "my_static_resource"
                    , 5
                    , "static_resource_url"
                    , "static_resource_checksum"
                    , 10
                )
            }
        )
    );

    auto result = factory.GetSingleClusterObjectManager(selectorResult, logger.SpawnFrame());
    UNIT_ASSERT(result);
    auto manager = result.Success();

    NController::ISingleClusterObjectManager::TDependentObjects dependentObjectsResult = {
        {
            MakeAtomicShared<NController::TSelectObjectsResult>(
                TVector<NController::TSelectorResultPtr>{
                    MakeAtomicShared<NYP::NClient::TSelectorResult>(GetPodAtributeList(POD_ID + "_1"))
                    , MakeAtomicShared<NYP::NClient::TSelectorResult>(GetPodAtributeList(POD_ID + "_2"))
                }
            )
        }  /* SelectedObjects */
        , {}  /* ObjectsAccessAllowedUserIds_ */
        , {}  /* GotObjects */
    };

    TVector<NYP::NClient::TCreateObjectRequest> create;
    TVector<NYP::NClient::TRemoveObjectRequest> remove;
    TVector<NYP::NClient::TUpdateRequest> update;
    NManagerFactoryTestLib::GenerateYpUpdates(manager, dependentObjectsResult, create, remove, update);

    UNIT_ASSERT(create.empty());
    UNIT_ASSERT(remove.empty());

    UNIT_ASSERT_EQUAL_C(update.size(), 3, update.size());

    // pod spec/resource_cache updates
    for (size_t i = 0; i < 2; ++i) {
        UNIT_ASSERT_EQUAL(update[i].GetObjectType(), NYP::NClient::NApi::NProto::OT_POD);
        UNIT_ASSERT_STRINGS_EQUAL(update[i].GetObjectId(), POD_ID + "_" + ToString(i + 1));

        UNIT_ASSERT_EQUAL_C(update[i].GetSetVec().size(), 1, update[i].GetSetVec().size());
        UNIT_ASSERT_STRINGS_EQUAL(update[i].GetSetVec()[0].GetPath(), "/spec/resource_cache");

        auto resourceCacheSpec = NManagerFactoryTestLib::ToJson(update[i].GetSetVec()[0].GetValue());
        UNIT_ASSERT(resourceCacheSpec["spec"]["layers"].IsArray());
        UNIT_ASSERT_EQUAL_C(resourceCacheSpec["spec"]["layers"].GetArraySafe().size(), 1, resourceCacheSpec["spec"]["layers"].GetArraySafe().size());
        UNIT_ASSERT(resourceCacheSpec["spec"]["static_resources"].IsArray());
        UNIT_ASSERT_EQUAL_C(resourceCacheSpec["spec"]["static_resources"].GetArraySafe().size(), 1, resourceCacheSpec["spec"]["static_resources"].GetArraySafe().size());
    }

    // resource_cache status update
    {
        UNIT_ASSERT_EQUAL(update[2].GetObjectType(), NYP::NClient::NApi::NProto::OT_RESOURCE_CACHE);
        UNIT_ASSERT_STRINGS_EQUAL(update[2].GetObjectId(), RESOURCE_CACHE_ID);

        UNIT_ASSERT_EQUAL_C(update[2].GetSetVec().size(), 1, update[2].GetSetVec().size());
        UNIT_ASSERT_STRINGS_EQUAL(update[2].GetSetVec()[0].GetPath(), "/status");

        auto resourceCacheStatus = NManagerFactoryTestLib::ToJson(update[2].GetSetVec()[0].GetValue());
        UNIT_ASSERT(resourceCacheStatus["cached_resource_status"].IsArray());
        UNIT_ASSERT_EQUAL_C(resourceCacheStatus["cached_resource_status"].GetArraySafe().size(), 2, resourceCacheStatus["cached_resource_status"].GetArraySafe().size());
    }
}

}

} // namespace NInfra::NResourceCacheController::NResourceCacheManagerTest
