#include "resource_cache_controller.h"

#include <infra/libs/sensors/macros.h>
#include <yp/cpp/yp/yson_interop.h>

#include <yt/yt/core/yson/public.h>

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/digest/md5/md5.h>

#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/string/builder.h>
#include <util/system/env.h>

namespace NInfra::NResourceCacheController {

namespace {

TString GetHashFromProto(const NProtoBuf::MessageLite& message) {
    return MD5::Calc(message.SerializeAsString());
}

TString GetHashFromResourceRevisionStatus(const NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceRevisionStatus& status) {
    if (status.has_layer()) {
        return "layer_" + GetHashFromProto(status.layer());
    } else if (status.has_static_resource()) {
        return "static_resource_" + GetHashFromProto(status.static_resource());
    }
    return "empty_hash";
}

void ApplyCachedResourceSpecToStatus(
    const NYP::NClient::NApi::NProto::TResourceCacheSpec::TCachedResource& cachedResourceSpec
    , NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceStatus& cachedResourceStatus
    , ui32 revision
) {
    NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceStatus newStatus;
    newStatus.set_id(cachedResourceStatus.id());
    NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceRevisionStatus* currentRevisionStatus = newStatus.mutable_revisions()->Add();

    THashSet<TString> resourceHashes;
    for (const auto& cachedResourceRevisionStatus : cachedResourceStatus.revisions()) {
        if ((cachedResourceSpec.has_layer() &&
            cachedResourceRevisionStatus.has_layer() &&
            google::protobuf::util::MessageDifferencer::Equals(cachedResourceSpec.layer(), cachedResourceRevisionStatus.layer()))
            ||
            (cachedResourceSpec.has_static_resource() &&
            cachedResourceRevisionStatus.has_static_resource() &&
            google::protobuf::util::MessageDifferencer::Equals(cachedResourceSpec.static_resource(), cachedResourceRevisionStatus.static_resource()))
        ) {
            currentRevisionStatus->CopyFrom(cachedResourceRevisionStatus);
        } else {
            if (!resourceHashes.contains(GetHashFromResourceRevisionStatus(cachedResourceRevisionStatus))) {
                newStatus.mutable_revisions()->Add()->CopyFrom(cachedResourceRevisionStatus);
                resourceHashes.insert(GetHashFromResourceRevisionStatus(cachedResourceRevisionStatus));
            }
        }
    }

    currentRevisionStatus->set_revision(revision);
    if (cachedResourceSpec.has_layer()) {
        currentRevisionStatus->mutable_layer()->CopyFrom(cachedResourceSpec.layer());
    } else if (cachedResourceSpec.has_static_resource()) {
        currentRevisionStatus->mutable_static_resource()->CopyFrom(cachedResourceSpec.static_resource());
    }

    while ((size_t)newStatus.revisions_size() > cachedResourceSpec.basic_strategy().max_latest_revisions()) {
        newStatus.mutable_revisions()->RemoveLast();
    }

    cachedResourceStatus.CopyFrom(newStatus);
}

google::protobuf::Timestamp ToTimestamp(const TInstant& time) {
    google::protobuf::Timestamp timestamp;
    timestamp.set_seconds(time.Seconds());
    timestamp.set_nanos(time.NanoSecondsOfSecond());
    return timestamp;
}

void AddConditionToAggregatedCondition(
    NYP::NClient::NApi::NProto::TAggregatedCondition& condition
    , const NInfra::NPodAgent::API::TCondition& conditionToAdd
) {
    if (conditionToAdd.status() == NInfra::NPodAgent::API::EConditionStatus_TRUE) {
        condition.set_pod_count(condition.pod_count() + 1);
    }
}

void IncrementAggregatedConditionPodCount(NYP::NClient::NApi::NProto::TAggregatedCondition& condition) {
    condition.set_pod_count(condition.pod_count() + 1);
}

void UpdateConditionStatus(
    NYP::NClient::NApi::NProto::TAggregatedCondition& condition
    , ui32 needCnt
    , NYP::NClient::NApi::NProto::EConditionStatus onNeed
    , NYP::NClient::NApi::NProto::EConditionStatus notNeed
    , const google::protobuf::Timestamp& now
) {
    const NYP::NClient::NApi::NProto::EConditionStatus oldStatus = condition.condition().status();

    if (condition.pod_count() == needCnt) {
        condition.mutable_condition()->set_status(onNeed);
    } else {
        condition.mutable_condition()->set_status(notNeed);
    }

    if (oldStatus != condition.condition().status()) {
        *condition.mutable_condition()->mutable_last_transition_time() = now;
    }
}

void ClearCondition(NYP::NClient::NApi::NProto::TAggregatedCondition& condition) {
    // Clear all except status and last_transition_time
    // because we need this fields for correct condition recalculation
    condition.mutable_condition()->clear_message();
    condition.mutable_condition()->clear_reason();
    condition.set_pod_count(0);
}

template<class TCachedResources, class TCachedResourceRevisionStatuses>
void AddCachedResourcesStatusToGlobalConditions(
    const TCachedResources& cachedResources
    , const TCachedResourceRevisionStatuses& cachedResourceRevisionStatuses
    , ui32 currentRevision
    , ui32& cntAllReady
    , bool& isAllInProgress
    , bool& isAllFailed
    , ui32& cntLatestReady
    , bool& isLatestInProgress
    , bool& isLatestFailed
) {
    for (const auto& cachedResource : cachedResources) {
        auto ptr = cachedResourceRevisionStatuses.FindPtr(std::make_pair(cachedResource.id(), cachedResource.revision()));

        if (ptr) {
            bool isLatest = (ptr->revision() == currentRevision);

            if (cachedResource.ready().status() == NPodAgent::API::EConditionStatus_TRUE) {
                ++cntAllReady;
                if (isLatest) {
                    ++cntLatestReady;
                }
            }
            if (cachedResource.in_progress().status() == NPodAgent::API::EConditionStatus_TRUE) {
                isAllInProgress = true;
                if (isLatest) {
                    isLatestInProgress = true;
                }
            }
            if (cachedResource.failed().status() == NPodAgent::API::EConditionStatus_TRUE) {
                isAllFailed = true;
                if (isLatest) {
                    isLatestFailed = true;
                }
            }
        }
    }
}

void CalcResourceCacheStatusGlobalConditions(
    NYP::NClient::NApi::NProto::TResourceCacheStatus& resourceCacheStatus
    , const TVector<NYP::NClient::TPod>& pods
    , const TMap<std::pair<TString, ui32>, NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceRevisionStatus&>& cachedLayerRevisionStatuses
    , const TMap<std::pair<TString, ui32>, NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceRevisionStatus&>& cachedStaticResourceRevisionStatuses
    , const google::protobuf::Timestamp& now
) {
    ClearCondition(*resourceCacheStatus.mutable_all_ready());
    ClearCondition(*resourceCacheStatus.mutable_all_in_progress());
    ClearCondition(*resourceCacheStatus.mutable_all_failed());

    ClearCondition(*resourceCacheStatus.mutable_latest_ready());
    ClearCondition(*resourceCacheStatus.mutable_latest_in_progress());
    ClearCondition(*resourceCacheStatus.mutable_latest_failed());

    ui32 needCntAllReady = 0;
    ui32 needCntLatestReady = 0;
    for (auto& cachedResourceStatus : *resourceCacheStatus.mutable_cached_resource_status()) {
        needCntAllReady += cachedResourceStatus.revisions().size();
        needCntLatestReady += !cachedResourceStatus.revisions().empty();
    }

    for (const auto& pod : pods) {
        ui32 cntAllReady = 0;
        bool isAllInProgress = false;
        bool isAllFailed = false;

        ui32 cntLatestReady = 0;
        bool isLatestInProgress = false;
        bool isLatestFailed = false;

        AddCachedResourcesStatusToGlobalConditions(
            pod.Status().agent().pod_agent_payload().status().resource_cache().layers()
            , cachedLayerRevisionStatuses
            , resourceCacheStatus.revision()
            , cntAllReady
            , isAllInProgress
            , isAllFailed
            , cntLatestReady
            , isLatestInProgress
            , isLatestFailed
        );

        AddCachedResourcesStatusToGlobalConditions(
            pod.Status().agent().pod_agent_payload().status().resource_cache().static_resources()
            , cachedStaticResourceRevisionStatuses
            , resourceCacheStatus.revision()
            , cntAllReady
            , isAllInProgress
            , isAllFailed
            , cntLatestReady
            , isLatestInProgress
            , isLatestFailed
        );

        if (cntAllReady == needCntAllReady) {
            IncrementAggregatedConditionPodCount(*resourceCacheStatus.mutable_all_ready());
        }
        if (isAllInProgress) {
            IncrementAggregatedConditionPodCount(*resourceCacheStatus.mutable_all_in_progress());
        }
        if (isAllFailed) {
            IncrementAggregatedConditionPodCount(*resourceCacheStatus.mutable_all_failed());
        }

        if (cntLatestReady == needCntLatestReady) {
            IncrementAggregatedConditionPodCount(*resourceCacheStatus.mutable_latest_ready());
        }
        if (isLatestInProgress) {
            IncrementAggregatedConditionPodCount(*resourceCacheStatus.mutable_latest_in_progress());
        }
        if (isLatestFailed) {
            IncrementAggregatedConditionPodCount(*resourceCacheStatus.mutable_latest_failed());
        }
    }

    constexpr auto conditionTrue = NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE;
    constexpr auto conditionFalse = NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE;

    UpdateConditionStatus(*resourceCacheStatus.mutable_all_ready(), pods.size(), conditionTrue, conditionFalse, now);
    UpdateConditionStatus(*resourceCacheStatus.mutable_all_in_progress(), 0, conditionFalse, conditionTrue, now);
    UpdateConditionStatus(*resourceCacheStatus.mutable_all_failed(), 0, conditionFalse, conditionTrue, now);

    UpdateConditionStatus(*resourceCacheStatus.mutable_latest_ready(), pods.size(), conditionTrue, conditionFalse, now);
    UpdateConditionStatus(*resourceCacheStatus.mutable_latest_in_progress(), 0, conditionFalse, conditionTrue, now);
    UpdateConditionStatus(*resourceCacheStatus.mutable_latest_failed(), 0, conditionFalse, conditionTrue, now);
}

void CalcResourceCacheStatusPerRevisionConditions(
    NYP::NClient::NApi::NProto::TResourceCacheStatus& resourceCacheStatus
    , const TVector<NYP::NClient::TPod>& pods
    , const TMap<std::pair<TString, ui32>, NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceRevisionStatus&>& cachedLayerRevisionStatuses
    , const TMap<std::pair<TString, ui32>, NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceRevisionStatus&>& cachedStaticResourceRevisionStatuses
    , const google::protobuf::Timestamp& now
) {
    for (auto& cachedResourceStatus : *resourceCacheStatus.mutable_cached_resource_status()) {
        for (auto& cachedResourceRevisionStatus : *cachedResourceStatus.mutable_revisions()) {
            ClearCondition(*cachedResourceRevisionStatus.mutable_ready());
            ClearCondition(*cachedResourceRevisionStatus.mutable_in_progress());
            ClearCondition(*cachedResourceRevisionStatus.mutable_failed());
        }
    }

    for (const auto& pod : pods) {
        for (const auto& cachedLayer : pod.Status().agent().pod_agent_payload().status().resource_cache().layers()) {
            auto ptr = cachedLayerRevisionStatuses.FindPtr(std::make_pair(cachedLayer.id(), cachedLayer.revision()));
            if (ptr) {
                AddConditionToAggregatedCondition(
                    *ptr->mutable_ready()
                    , cachedLayer.ready()
                );
                AddConditionToAggregatedCondition(
                    *ptr->mutable_in_progress()
                    , cachedLayer.in_progress()
                );
                AddConditionToAggregatedCondition(
                    *ptr->mutable_failed()
                    , cachedLayer.failed()
                );
            }
        }
        for (const auto& cachedStaticResource : pod.Status().agent().pod_agent_payload().status().resource_cache().static_resources()) {
            auto ptr = cachedStaticResourceRevisionStatuses.FindPtr(std::make_pair(cachedStaticResource.id(), cachedStaticResource.revision()));
            if (ptr) {
                AddConditionToAggregatedCondition(
                    *ptr->mutable_ready()
                    , cachedStaticResource.ready()
                );
                AddConditionToAggregatedCondition(
                    *ptr->mutable_in_progress()
                    , cachedStaticResource.in_progress()
                );
                AddConditionToAggregatedCondition(
                    *ptr->mutable_failed()
                    , cachedStaticResource.failed()
                );
            }
        }
    }

    for (auto& cachedResourceStatus : *resourceCacheStatus.mutable_cached_resource_status()) {
        for (auto& cachedResourceRevisionStatus : *cachedResourceStatus.mutable_revisions()) {
            constexpr auto conditionTrue = NYP::NClient::NApi::NProto::EConditionStatus::CS_TRUE;
            constexpr auto conditionFalse = NYP::NClient::NApi::NProto::EConditionStatus::CS_FALSE;

            UpdateConditionStatus(*cachedResourceRevisionStatus.mutable_ready(), pods.size(), conditionTrue, conditionFalse, now);
            UpdateConditionStatus(*cachedResourceRevisionStatus.mutable_in_progress(), 0, conditionFalse, conditionTrue, now);
            UpdateConditionStatus(*cachedResourceRevisionStatus.mutable_failed(), 0, conditionFalse, conditionTrue, now);
        }
    }
}

bool CheckIsPodReady(const NYP::NClient::TPod& pod) {
    TSet<std::pair<ui32, TString>> specLayerSet;
    TSet<std::pair<ui32, TString>> specStaticResourceSet;
    TSet<std::pair<ui32, TString>> statusLayerSet;
    TSet<std::pair<ui32, TString>> statusStaticResourceSet;

    for (const auto& layerStatus : pod.Status().agent().pod_agent_payload().status().resource_cache().layers()) {
        if (layerStatus.ready().status() != NInfra::NPodAgent::API::EConditionStatus_TRUE) {
            return false;
        }
        statusLayerSet.insert({layerStatus.revision(), layerStatus.id()});
    }
    for (const auto& staticResourceStatus : pod.Status().agent().pod_agent_payload().status().resource_cache().static_resources()) {
        if (staticResourceStatus.ready().status() != NInfra::NPodAgent::API::EConditionStatus_TRUE) {
            return false;
        }
        statusStaticResourceSet.insert({staticResourceStatus.revision(), staticResourceStatus.id()});
    }

    for (const auto& layerSpec : pod.Spec().resource_cache().spec().layers()) {
        specLayerSet.insert({layerSpec.revision(), layerSpec.layer().id()});
    }
    for (const auto& staticResourceSpec : pod.Spec().resource_cache().spec().static_resources()) {
        specStaticResourceSet.insert({staticResourceSpec.revision(), staticResourceSpec.resource().id()});
    }

    return specLayerSet == statusLayerSet && specStaticResourceSet == statusStaticResourceSet;
}

void UpdatePodSpecs(
    const TVector<NYP::NClient::TPod>& pods
    , const NYP::NClient::NApi::NProto::TPodSpec::TPodAgentResourceCache& podAgentResourceCache
    , const ui32 updateWindow
    , TVector<NYP::NClient::TUpdateRequest>& updates
) {
    TVector<TString> podsIdCurrentSpec;
    TVector<TString> podsIdOldSpec;
    TVector<TString> podsIdForUpdate;

    ui32 numberOfNotReadyPods = 0;
    for (const auto& pod : pods) {
        bool isCurrentSpec = google::protobuf::util::MessageDifferencer::Equals(pod.Spec().resource_cache(), podAgentResourceCache);
        if (CheckIsPodReady(pod)) {
            if (isCurrentSpec) {
                podsIdCurrentSpec.push_back(pod.Meta().id());
            } else {
                podsIdOldSpec.push_back(pod.Meta().id());
            }
        } else {
            if (!isCurrentSpec) {
                podsIdForUpdate.push_back(pod.Meta().id());
            }
            ++numberOfNotReadyPods;
        }
    }

    if (updateWindow > numberOfNotReadyPods || updateWindow == 0) {
        // if updateWindow is equal to zero then it's considered equal to infinity for backward compatibility
        ui32 rem = updateWindow == 0 ? podsIdOldSpec.size() : updateWindow - numberOfNotReadyPods;

        Sort(podsIdOldSpec.begin(), podsIdOldSpec.end());

        for (ui32 i = 0; i < rem && i < podsIdOldSpec.size(); ++i) {
            podsIdForUpdate.push_back(podsIdOldSpec[i]);
        }
    }

    // This sorting for the stable order of updates
    Sort(podsIdForUpdate.begin(), podsIdForUpdate.end());

    for (const auto& podId : podsIdForUpdate) {
        updates.emplace_back(
            NYP::NClient::NApi::NProto::EObjectType::OT_POD
            , podId
            , TVector<NYP::NClient::TSetRequest>{NYP::NClient::TSetRequest("/spec/resource_cache", podAgentResourceCache, true)}
            , TVector<NYP::NClient::TRemoveRequest>{}
        );
    }
}

void UpdateResourceCacheStatus(
    const NYP::NClient::TResourceCache& currentResourceCache
    , const NYP::NClient::TResourceCache& lastLoadedResourceCache
    , TVector<NYP::NClient::TUpdateRequest>& updates
) {
    if (!google::protobuf::util::MessageDifferencer::Equals(currentResourceCache.Status(), lastLoadedResourceCache.Status())) {
        updates.emplace_back(
            NYP::NClient::NApi::NProto::EObjectType::OT_RESOURCE_CACHE
            , currentResourceCache.Meta().id()
            , TVector<NYP::NClient::TSetRequest>{NYP::NClient::TSetRequest("/status", currentResourceCache.Status(), true)}
            , TVector<NYP::NClient::TRemoveRequest>{}
        );
    }
}

} // namespace

void TResourceCacheController::ApplySpecToCurrentStatus() {
    TMap<TString, const NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceStatus> cachedResourceStatuses;
    for (const auto& cachedResourceStatus : CurrentResourceCache_.Status().cached_resource_status()) {
        cachedResourceStatuses.insert({cachedResourceStatus.id(), cachedResourceStatus});
    }

    NYP::NClient::NApi::NProto::TResourceCacheStatus newStatus;
    newStatus.set_revision(CurrentResourceCache_.Spec().revision());
    for (const auto& cachedResource : CurrentResourceCache_.Spec().cached_resources()) {
        auto* cachedResourceStatus = newStatus.add_cached_resource_status();
        if (cachedResourceStatuses.contains(cachedResource.id())) {
            cachedResourceStatus->CopyFrom(cachedResourceStatuses.at(cachedResource.id()));
        } else {
            cachedResourceStatus->set_id(cachedResource.id());
        }

        ApplyCachedResourceSpecToStatus(cachedResource, *cachedResourceStatus, CurrentResourceCache_.Spec().revision());
    }

    CurrentResourceCache_.MutableStatus()->CopyFrom(newStatus);
}

void TResourceCacheController::AddPodsStatusToCurrentStatus(const TVector<NYP::NClient::TPod>& pods) {
    TMap<std::pair<TString, ui32>, NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceRevisionStatus&> cachedLayerRevisionStatuses;
    TMap<std::pair<TString, ui32>, NYP::NClient::NApi::NProto::TResourceCacheStatus::TCachedResourceRevisionStatus&> cachedStaticResourceRevisionStatuses;

    for (auto& cachedResourceStatus : *CurrentResourceCache_.MutableStatus()->mutable_cached_resource_status()) {
        for (auto& cachedResourceRevisionStatus : *cachedResourceStatus.mutable_revisions()) {
            if (cachedResourceRevisionStatus.has_layer()) {
                cachedLayerRevisionStatuses.insert({{cachedResourceStatus.id(), cachedResourceRevisionStatus.revision()}, cachedResourceRevisionStatus});
            } else if (cachedResourceRevisionStatus.has_static_resource()) {
                cachedStaticResourceRevisionStatuses.insert({{cachedResourceStatus.id(), cachedResourceRevisionStatus.revision()}, cachedResourceRevisionStatus});
            }
        }
    }

    const auto now = ToTimestamp(TInstant::Now());

    CalcResourceCacheStatusPerRevisionConditions(
        *CurrentResourceCache_.MutableStatus()
        , pods
        , cachedLayerRevisionStatuses
        , cachedStaticResourceRevisionStatuses
        , now
    );

    CalcResourceCacheStatusGlobalConditions(
        *CurrentResourceCache_.MutableStatus()
        , pods
        , cachedLayerRevisionStatuses
        , cachedStaticResourceRevisionStatuses
        , now
    );
}

NInfra::NPodAgent::API::TPodAgentResourceCacheSpec TResourceCacheController::GeneratePodAgentResourceCacheSpec() const {
    // To sort resources by pair <id, revision>
    TMap<std::pair<TString, ui32>, NInfra::NPodAgent::API::TCacheLayer> cacheLayers;
    TMap<std::pair<TString, ui32>, NInfra::NPodAgent::API::TCacheResource> cacheStaticResources;

    for (const auto& cachedResourceStatus : CurrentResourceCache_.Status().cached_resource_status()) {
        for (const auto& cachedResourceRevisionStatus : cachedResourceStatus.revisions()) {
            if (cachedResourceRevisionStatus.has_layer()) {
                NInfra::NPodAgent::API::TCacheLayer cacheLayer;
                cacheLayer.mutable_layer()->CopyFrom(cachedResourceRevisionStatus.layer());
                cacheLayer.mutable_layer()->set_id(cachedResourceStatus.id());
                cacheLayer.set_revision(cachedResourceRevisionStatus.revision());

                cacheLayers.insert({{cachedResourceStatus.id(), cachedResourceRevisionStatus.revision()}, cacheLayer});
            } else if (cachedResourceRevisionStatus.has_static_resource()) {
                NInfra::NPodAgent::API::TCacheResource cacheStaticResource;
                cacheStaticResource.mutable_resource()->CopyFrom(cachedResourceRevisionStatus.static_resource());
                cacheStaticResource.mutable_resource()->set_id(cachedResourceStatus.id());
                cacheStaticResource.set_revision(cachedResourceRevisionStatus.revision());

                cacheStaticResources.insert({{cachedResourceStatus.id(), cachedResourceRevisionStatus.revision()}, cacheStaticResource});
            }
        }
    }

    NInfra::NPodAgent::API::TPodAgentResourceCacheSpec spec;

    for (const auto& cacheLayer : cacheLayers) {
        spec.mutable_layers()->Add()->CopyFrom(cacheLayer.second);
    }
    for (const auto& cacheStaticResource : cacheStaticResources) {
        spec.mutable_static_resources()->Add()->CopyFrom(cacheStaticResource.second);
    }

    return spec;
}

TVector<NYP::NClient::TUpdateRequest> TResourceCacheController::GenerateUpdates(const TVector<NYP::NClient::TPod>& pods) {
    NYP::NClient::NApi::NProto::TPodSpec::TPodAgentResourceCache podAgentResourceCache;
    podAgentResourceCache.mutable_spec()->CopyFrom(GeneratePodAgentResourceCacheSpec());

    TVector<NYP::NClient::TUpdateRequest> updates;

    UpdatePodSpecs(
        pods
        , podAgentResourceCache
        , CurrentResourceCache_.Spec().update_window()
        , updates
    );

    UpdateResourceCacheStatus(
        CurrentResourceCache_
        , LastLoadedResourceCache_
        , updates
    );

    return updates;
}

} // namespace NInfra::NResourceCacheController
