from __future__ import unicode_literals
import os
import logging
import logging.config

import flask
import gevent
import raven.conf
import raven.handlers.logging
from gevent import pywsgi
import yp.client
from sepelib.core import config
from infra.swatlib.zk import client
from infra.rsc.src import test
from infra.rsc.src.lib import podutil
from infra.rsc.src.model import consts
from infra.rsc.src.model import gc
from infra.rsc.src.model import manager
from infra.rsc.src.model import pod_maker
from infra.rsc.src.model import rate_limiter
from infra.rsc.src.model import reflector
from infra.rsc.src.model import rs_updater
from infra.rsc.src.model import runner
from infra.rsc.src.model import storage
from infra.rsc.src.model import yp_client
from infra.controllers_monitoring.src.lib import sockutil
from infra.swatlib.gevent import exclusiveservice
from infra.swatlib import metrics

log = logging.getLogger('rsc')


class Application(object):
    """
    God object, managing service lifetime.
    """
    name = 'replica_set_controller'

    @staticmethod
    def setup_logging(sentry_dsn=None):
        if not sentry_dsn:
            return
        c = raven.Client(dsn=sentry_dsn)
        c.transport_options['maximum_outstanding_requests'] = 50
        handler = raven.handlers.logging.SentryHandler(client,
                                                       level=logging.WARN)
        raven.conf.setup_logging(handler)

    @staticmethod
    def init_http_server(config, metrics_registry):
        app = flask.Flask('mcrsc')
        app.add_url_rule('/ping', view_func=lambda: '')

        metrics.MetricsExt(app, registry=metrics_registry, cfg={
            'yasm_stat_url': '/stat',
            'performance_enabled': False,
            'status_code_enabled': False,
            'export_current_timestamp': False

        })
        c = config.get_value('web').get('http')
        host = c.get('host', '::')
        port = c['port']

        log.info('Binding [%s]:%d for HTTP server...', host, port)
        sock, err = sockutil.create_server_socket(host, port)
        if err is not None:
            log.error('Failed to init server socket: %s', err)
            raise SystemExit(1)
        # Use gevent http server
        return pywsgi.WSGIServer(sock, app)

    def __init__(self, instance_id):
        sentry_dsn = config.get_value('sentry.dsn', default=None)
        self.setup_logging(sentry_dsn=sentry_dsn)
        # Init coordination service client.
        # Several instances can run on one host, use port for distinction.
        self.coord = client.ZookeeperClient(
            cfg={
                'hosts': config.get_value('coord.hosts'),
                'zk_root': config.get_value('coord.root'),
                'read_only': False,
                'log_debug': config.get_value('coord.log_debug'),
            },
            identifier=instance_id,
        )

        # Instantiate yp client.
        c = config.get_value('yp')
        token = c.get('token')
        if not token:
            c['token'] = os.environ.get('YP_TOKEN')
        address = c['address']
        client_base = yp.client.YpClient(address=address, config=c)
        stub = client_base.create_grpc_object_stub()
        yp_rsc_client = yp_client.YpClient(stub=stub)

        reg = metrics.Registry()

        # Instantiate reflectors.
        batch_size = config.get_value('reflector.batch_size')
        sleep_secs = config.get_value('reflector.sleep_secs')
        rs_match_dict = config.get_value('reflector.rs_match_labels',
                                         default=None)
        pod_match_dict = config.get_value('reflector.pod_match_labels')

        # TODO: get rid of this ugly deploy_engine hack after removing
        # __env__* replica_sets.
        rs_filter = (
            '[/labels/deploy_engine] = "{}" OR '
            '[/labels/deploy_engine] = null'
        ).format(consts.DEPLOY_ENGINE)

        if rs_match_dict:
            rs_match_dict.pop('deploy_engine', None)
            rs_filter = '({}) AND ({})'.format(
                rs_filter,
                podutil.make_filter_from_dict(rs_match_dict)
            )

        pod_match_dict['deploy_engine'] = consts.DEPLOY_ENGINE
        pod_filter = podutil.make_filter_from_dict(pod_match_dict)

        rs_storage = storage.IndexedStorage()
        ps_storage = storage.IndexedStorage()
        pod_storage = storage.PodIndexedStorage()
        runner_q = gevent.queue.Queue(maxsize=1)
        cluster = config.get_value('yp.cluster')
        rs_reflector = reflector.Reflector(name='rs_reflector',
                                           cluster=cluster,
                                           obj_type=yp.data_model.OT_REPLICA_SET,
                                           obj_class=yp.data_model.TReplicaSet,
                                           obj_filter=rs_filter,
                                           storage=rs_storage,
                                           client=yp_rsc_client,
                                           batch_size=batch_size,
                                           sleep_secs=sleep_secs,
                                           metrics_registry=reg)
        ps_reflector = reflector.Reflector(name='pod_set_reflector',
                                           cluster=cluster,
                                           obj_type=yp.data_model.OT_POD_SET,
                                           obj_class=yp.data_model.TPodSet,
                                           obj_filter=pod_filter,
                                           storage=ps_storage,
                                           client=yp_rsc_client,
                                           batch_size=batch_size,
                                           sleep_secs=sleep_secs,
                                           metrics_registry=reg)
        pod_reflector = reflector.Reflector(name='pod_reflector',
                                            cluster=cluster,
                                            obj_type=yp.data_model.OT_POD,
                                            obj_class=yp.data_model.TPod,
                                            obj_filter=pod_filter,
                                            storage=pod_storage,
                                            client=yp_rsc_client,
                                            batch_size=batch_size,
                                            sleep_secs=sleep_secs,
                                            runner_q=runner_q,
                                            metrics_registry=reg)

        # Instantiate pod gc.
        gc_sleep_secs = config.get_value('gc.sleep_secs')
        pod_set_gc = gc.PodSetGc(name="pod_set_gc",
                                 client=yp_rsc_client,
                                 rs_storage=rs_storage,
                                 pod_set_storage=ps_storage,
                                 sleep_secs=gc_sleep_secs)

        # Instantiate ctl.
        is_inplace_update_disabled = config.get_value(
            'yp.disable_inplace_update',
            False
        )
        use_deploy_status = config.get_value('use_deploy_status', False)

        delay_secs = config.get_value(
            'rate_limiter.delay_secs',
            consts.DEFAULT_RATE_LIMITER_DELAY_SECS
        )
        rate_lmtr = rate_limiter.RateLimiter(
            config.get_value('rate_limiter.enabled', True),
            delay_secs
        )
        root_users = config.get_value('yp.root_users')
        pod_match_dict['deploy_engine'] = consts.DEPLOY_ENGINE
        pod_match_labels = podutil.make_labels_from_dict(pod_match_dict)

        maker = pod_maker.PodMaker(root_users=root_users,
                                   match_labels=pod_match_labels)
        ctl = rs_updater.ReplicaSetPodsUpdater(
            client=yp_rsc_client,
            match_labels=pod_match_labels,
            is_inplace_update_disabled=is_inplace_update_disabled,
            use_deploy_status=use_deploy_status,
            pod_maker=maker,
            rate_limiter=rate_lmtr
        )

        # Instantiate ctl runner.
        threads_count = config.get_value('runner.threads_count')
        ctl_runner = runner.CtlRunner(ctl=ctl,
                                      q=runner_q,
                                      cluster=cluster,
                                      rs_storage=rs_storage,
                                      pod_storage=pod_storage,
                                      threads_count=threads_count,
                                      metrics_registry=reg)

        # Instantiate rsc manager.
        mngr = manager.RscManager(rs_reflector=rs_reflector,
                                  pod_set_reflector=ps_reflector,
                                  pod_reflector=pod_reflector,
                                  ctl_runner=ctl_runner,
                                  pod_set_gc=pod_set_gc)
        self.manager = exclusiveservice.ExclusiveService(self.coord, 'rsc', mngr)

        # Instantiate http server for stats.
        self.srv = self.init_http_server(config, metrics_registry=reg)

    # Public methods
    @staticmethod
    def setup_environment():
        # Patch requests connection pool to use gevent queue
        from requests.packages.urllib3.connectionpool import ConnectionPool
        from gevent.queue import LifoQueue

        ConnectionPool.QueueCls = LifoQueue
        # Disable requests spamming about:
        # Connection pool is full, discarding connection
        # This is the way we use alemate-http to avoid broken connections
        # There is nothing we can do about it, so simply mute
        logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.ERROR)

    def teardown_environment(self):
        pass

    def run(self):
        """
        Start application.
        Blocks until stop was called.
        """
        log.info('starting service...')
        self.setup_environment()
        # Run http server in separate thread.
        gevent.spawn(self.srv.serve_forever)

        self.coord.start().wait()
        self.manager.start()
        if config.get_value('debug.create_test_rs', False):
            log.info('creating test RS')
            test.recreate_rs()
        self.manager.wait()

    def stop(self):
        """
        Gracefully stop application.
        Can block for a long time or throw exception, be ready.
        """
        log.info('stopping service...')
        self.manager.stop()
        # self.coord.stop()
        self.teardown_environment()
        log.info('=' * 30)
