import random
import string

import yt.yson as yson
import yp.data_model
from yp_proto.yp.client.api.proto import pod_agent_pb2
from yp_proto.yp.client.api.proto import conditions_pb2
from infra.swatlib import randomutil
from sepelib.core import config


# TODO: remove PCS_START_PENDING from AGENT_ALIVE_STATES
# after fixing https://st.yandex-team.ru/ISS-6019.
AGENT_ALIVE_STATES = (yp.data_model.PCS_STARTED,
                      yp.data_model.PCS_START_PENDING)


EVICTION_ALIVE_STATES = (yp.data_model.ES_NONE, yp.data_model.ES_REQUESTED)

OBJ_ID_LETTERS = string.ascii_lowercase + string.digits


def is_pod_alive(pod):
    """
    :type pod: yp.data_model.TPod
    :rtype: bool
    """
    if not pod.status.scheduling.node_id:
        return False
    if pod.status.eviction.state not in EVICTION_ALIVE_STATES:
        return False
    if pod.status.agent.pod_agent_payload.status.revision != pod.spec.pod_agent_payload.spec.revision:
        return False
    if pod.status.agent.state not in AGENT_ALIVE_STATES:
        return False
    if pod.status.agent.pod_agent_payload.status.ready.status == pod_agent_pb2.EConditionStatus_TRUE:
        return True
    return False


def is_pod_finally_dead(pod):
    """
    :type pod: yp.data_model.TPod
    :rtype: bool
    """
    return pod.status.eviction.state == yp.data_model.ES_REQUESTED


def increment_pod_count(status, is_ready):
    """
    :type is_ready: bool
    """
    if is_ready:
        target = status.ready
    else:
        target = status.in_progress
    target.pod_count += 1
    target.condition.last_transition_time.GetCurrentTime()


def set_pod_count(status, amount):
    """
    :type status: yp.data_model.TReplicaSetStatus.TAggregatedCondition
    :type amount: int
    """
    status.pod_count = amount
    status.condition.last_transition_time.GetCurrentTime()


def update_rs_status(rs, pods, ready_count, in_progress_count):
    """
    :type rs: yp.data_model.TReplicaSet
    :type pods: list[yp.data_model.TPod]
    :type ready_count: int
    :type in_progress_count: int
    :rtype: yp.data_model.TReplicaSet
    """
    rs.status.revision_id = rs.spec.revision_id
    set_pod_count(rs.status.ready, amount=ready_count)
    set_pod_count(rs.status.in_progress, amount=in_progress_count)
    min_ready = ready_count - rs.spec.deployment_strategy.max_unavailable
    if ready_count >= min_ready:
        rs.status.ready.condition.status = 'True'
        rs.status.ready_condition.status = conditions_pb2.CS_TRUE
        rs.status.in_progress.condition.status = 'False'
        rs.status.in_progress_condition.status = conditions_pb2.CS_FALSE
    else:
        rs.status.ready.condition.status = 'False'
        rs.status.ready_condition.status = conditions_pb2.CS_FALSE
        rs.status.in_progress.condition.status = 'True'
        rs.status.in_progress_condition.status = conditions_pb2.CS_TRUE
    rs.status.revisions.clear()
    for pod in pods:
        pod_rev = str(pod.spec.pod_agent_payload.spec.revision)
        agent_status = pod.status.agent.pod_agent_payload
        rs.status.revisions[pod_rev].revision_id = pod_rev

        increment_pod_count(rs.status.revisions[pod_rev], is_ready=is_pod_alive(pod))

        for volume in agent_status.status.volumes:
            v = rs.status.revisions[pod_rev].volumes.add()
            v.id = volume.id
            increment_pod_count(v, is_ready=volume.state == pod_agent_pb2.EVolumeState_READY)

        for workload in agent_status.status.workloads:
            w = rs.status.revisions[pod_rev].workloads.add()
            w.id = workload.id
            increment_pod_count(w, is_ready=workload.state == pod_agent_pb2.EWorkloadState_ACTIVE)

        for box in agent_status.status.boxes:
            b = rs.status.revisions[pod_rev].boxes.add()
            b.id = box.id
            increment_pod_count(b, is_ready=box.state == pod_agent_pb2.EBoxState_READY)
    return rs


def is_workload_updateable(src, target):
    """
    :type src: yp_proto.yp.client.api.proto.pod_agent_pb2.TWorkload
    :type target: yp_proto.yp.client.api.proto.pod_agent_pb2.TWorkload
    :rtype: bool
    """
    # NOTE: maybe it would be better to compare max init resources
    for s, t in zip(src.init, target.init):
        if s.compute_resources != t.compute_resources:
            return False
    return src.start.compute_resources == target.start.compute_resources


def is_box_updateable(src, target):
    """
    :type src: yp_proto.yp.client.api.proto.pod_agent_pb2.TBox
    :type target: yp_proto.yp.client.api.proto.pod_agent_pb2.TBox
    :rtype: bool
    """
    # NOTE: maybe it would be better to compare max init resources
    for s, t in zip(src.init, target.init):
        if s.compute_resources != t.compute_resources:
            return False
    return True


def is_resource_requests_updateable(spec, template):
    # DEPLOY-1722: we must compare every single field
    # to avoid case when YP sets some field value on pod saving
    if spec.vcpu_guarantee != template.vcpu_guarantee:
        return False
    if spec.vcpu_limit != template.vcpu_limit:
        return False

    if spec.memory_guarantee != template.memory_guarantee:
        return False
    if spec.memory_limit != template.memory_limit:
        return False
    if spec.anonymous_memory_limit != template.anonymous_memory_limit:
        return False
    if spec.dirty_memory_limit != template.dirty_memory_limit:
        return False

    if spec.network_bandwidth_guarantee != template.network_bandwidth_guarantee:
        return False
    if spec.network_bandwidth_limit != template.network_bandwidth_limit:
        return False
    return True


def is_pod_spec_updateable(spec, template):
    """
    :type src: yp.data_model.TPodSpec
    :type target: yp.data_model.TPodSpec
    :rtype: bool
    """
    if not is_resource_requests_updateable(spec.resource_requests, template.resource_requests):
        return False

    if spec.disk_volume_requests != template.disk_volume_requests:
        return False

    if spec.ip6_address_requests != template.ip6_address_requests:
        return False

    if spec.host_devices != template.host_devices:
        return False

    if config.get_value("inplace_update_compare_workloads", True):
        # Compare workloads.
        src_workloads = spec.pod_agent_payload.spec.workloads
        target_workloads = template.pod_agent_payload.spec.workloads
        if len(src_workloads) != len(target_workloads):
            return False

        d = {w.id: w for w in target_workloads}
        for s in src_workloads:
            t = d.get(s.id)
            if not t or not is_workload_updateable(s, t):
                return False

        # Compare boxes.
        src_boxes = spec.pod_agent_payload.spec.boxes
        target_boxes = template.pod_agent_payload.spec.boxes
        if len(src_boxes) != len(target_boxes):
            return False

        d = {b.id: b for b in target_boxes}
        for s in src_boxes:
            t = d.get(s.id)
            if not t or not is_box_updateable(s, t):
                return False

    inplace_update_enabled = config.get_value("enable_pod_agent_in_place_update", False)

    if not inplace_update_enabled and spec.pod_agent_payload.meta != template.pod_agent_payload.meta:
        return False

    return True


def get_label(labels, key, default=None):
    """
    :type labels: yp.data_model.TAttributeDictionary
    :type key: unicode
    :type default: unicode | None
    :rtype: unicode | None
    """
    for a in labels.attributes:
        if a.key == key:
            return yson.loads(a.value)
    return default


def set_label(labels, key, value):
    """
    :type labels: yp.data_model.TAttributeDictionary
    :type key: unicode
    :type value: unicode
    """
    attr = None
    for a in labels.attributes:
        if a.key == key:
            attr = a
            break
    if attr is None:
        attr = labels.attributes.add()
        attr.key = key
    attr.value = yson.dumps(value)


def make_labels_from_dict(d):
    """
    :type d: dict[str, str]
    :rtype: yp.data_model.TAttributeDictionary
    """
    rv = yp.data_model.TAttributeDictionary()
    for k, v in d.iteritems():
        if v is not None:
            set_label(rv, k, v)
    return rv


def make_dict_from_labels(labels):
    """
    :type labels: yp.data_model.TAttributeDictionary
    :rtype: dict[str, str]
    """
    d = {}
    for a in labels.attributes:
        v = yson.loads(a.value)
        if v is not None:
            d[a.key] = yson.loads(a.value)
    return d


def merge_labels(labels1, labels2):
    d1 = make_dict_from_labels(labels1)
    d2 = make_dict_from_labels(labels2)
    for k, v in d2.iteritems():
        d1[k] = v
    return make_labels_from_dict(d1)


def get_stage_id_from_rs(rs):
    stage_id = get_label(rs.labels, 'stage_id')
    if stage_id:
        return stage_id
    return rs.meta.id.split('.')[0]


def get_stage_id_from_pod_id(pod_id):
    parts = pod_id.rsplit('-', 1)
    if len(parts) < 2:
        return
    return parts[0]


def get_replica_num_from_pod_id(pod_id):
    parts = pod_id.rsplit('-', 1)
    if len(parts) < 2:
        # Ignore old-style pod ids
        return
    try:
        return int(parts[-1])
    except ValueError:
        # Some unexpected pod id pattern, ignore it
        pass


def make_filter_from_dict(d):
    """
    :type d: dict[str, str]
    :rtype: str
    """
    parts = []
    for k, v in d.iteritems():
        if v is None:
            p = '[/labels/{}] = null'.format(k)
        else:
            p = '[/labels/{}] = "{}"'.format(k, v)
        parts.append(p)
    return ' AND '.join(parts)


def gen_pod_id():
    return '{}{}'.format(random.choice(string.ascii_lowercase),
                         randomutil.gen_random_str(bits=80)[1:])


def stringify_revisions(revisions):
    """
    :type revisions: dict[str, yp_proto.yp.client.api.proto.deploy_pb2.TDeployProgress]
    :rtype: str
    """
    parts = []
    for r, p in revisions.iteritems():
        part = "rev={}: in_progress={}/ready={}".format(r,
                                                        p.pods_in_progress,
                                                        p.pods_ready)
        parts.append(part)
    return ', '.join(parts)


def update_pod_set_needed(ps, template):
    if ps.spec.account_id != template.spec.account_id:
        return True
    if ps.spec.antiaffinity_constraints != template.spec.antiaffinity_constraints:
        return True
    if ps.meta.acl != template.meta.acl:
        return True
    return False
