from __future__ import unicode_literals

import yp.data_model as data_model

from infra.swatlib.gevent import geventutil as gutil


def increment_progress(p, ready=0, in_progress=0):
    """
    :type p: yp_proto.yp.client.api.proto.deploy_pb2.TDeployProgress
    :type ready: int
    :type in_progress: int
    """
    p.pods_ready += ready
    p.pods_in_progress += in_progress
    p.pods_total += ready + in_progress


def increment_revision_progress(s, revision, target, ready, in_progress,
                                use_deploy_status):
    """
    :type s: yp.data_model.TReplicaSetStatus | yp_proto.yp.client.api.proto.deploy_pb2.TDeployStatus
    :type revision: int
    :type target: int
    :type ready: int
    :type in_progress: int
    :type use_deploy_status: bool
    """
    if use_deploy_status:
        increment_progress(s.revisions[revision], ready, in_progress)
    else:
        increment_progress(s.revisions_progress[str(revision)],
                           ready,
                           in_progress)
    increment_progress(s.total_progress, ready, in_progress)
    if revision == target:
        increment_progress(s.current_revision_progress,
                           ready,
                           in_progress)


def fill_deploy_status(s, cur_state, target, ctl_error, replica_count, use_deploy_status):
    """
    :type s: yp.data_model.TReplicaSetStatus | yp_proto.yp.client.api.proto.deploy_pb2.TDeployStatus
    :type cur_state: infra.rsc.src.model.state.CurrentState
    :type target: int
    :type ctl_error: Exception
    :type use_deploy_status: bool
    """
    for r in gutil.gevent_idle_iter(cur_state.ready):
        c = cur_state.ready.count(r)
        increment_revision_progress(s, r, target,
                                    ready=c,
                                    in_progress=0,
                                    use_deploy_status=use_deploy_status)

    for r in gutil.gevent_idle_iter(cur_state.in_progress):
        c = cur_state.in_progress.count(r)
        increment_revision_progress(s, r, target,
                                    ready=0,
                                    in_progress=c,
                                    use_deploy_status=use_deploy_status)

    ctl_status = s.controller_status
    ctl_status.Clear()
    fill_controller_status_succeeded(ctl_status.last_attempt.succeeded,
                                     ctl_error,
                                     use_deploy_status)
    # For current_revision we set pods_total to replica_count:
    # https://st.yandex-team.ru/DEPLOY-524
    s.current_revision_progress.pods_total = replica_count


def fill_controller_status_succeeded(s, ctl_error, use_deploy_status):
    """
    :type s: yp_proto.yp.client.api.proto.conditions_pb2.TCondition
    ;type ctl_error: Exception
    :type use_deploy_status: bool
    """
    s.last_transition_time.GetCurrentTime()
    if use_deploy_status:
        if ctl_error:
            s.status = data_model.CS_FALSE
            s.message = str(ctl_error)
        else:
            s.status = data_model.CS_TRUE
    else:
        if ctl_error:
            s.status = 'FALSE'
            s.message = str(ctl_error)
        else:
            s.status = 'TRUE'


def fill_status_ready_in_progress(s, replica_count, deployment_strategy,
                                  current_revision_progress):
    """
    :type s: yp.data_model.TReplicaSetStatus
    ;type replica_count: int
    :type deployment_strategy: yp.data_model.TReplicaSet.TDeploymentStrategy
    :type current_revision_progress: yp_proto.yp.client.api.proto.deploy_pb2.TDeployProgress
    """
    min_ready = replica_count - deployment_strategy.max_unavailable
    is_ready = (current_revision_progress.pods_ready >= min_ready and
                current_revision_progress.pods_ready > 0)
    if is_ready:
        s.ready_condition.status = data_model.CS_TRUE
        s.in_progress_condition.status = data_model.CS_FALSE
    else:
        s.ready_condition.status = data_model.CS_FALSE
        s.in_progress_condition.status = data_model.CS_TRUE

    ts = s.ready_condition.last_transition_time
    ts.GetCurrentTime()
    s.in_progress_condition.last_transition_time.CopyFrom(ts)


def set_condition_timestamp(new_condition, old_condition, ts):
    is_equal = (new_condition.status == old_condition.status and
                new_condition.reason == old_condition.reason and
                new_condition.message == old_condition.message)
    if is_equal:
        new_condition.last_transition_time.CopyFrom(
            old_condition.last_transition_time
        )
    else:
        new_condition.last_transition_time.CopyFrom(ts)


def set_status_timestamps(new_status, old_status):
    new_status.ready_condition.last_transition_time.GetCurrentTime()
    ts = new_status.ready_condition.last_transition_time

    set_condition_timestamp(new_status.ready_condition,
                            old_status.ready_condition,
                            ts)
    set_condition_timestamp(new_status.in_progress_condition,
                            old_status.in_progress_condition,
                            ts)
    set_condition_timestamp(
        new_status.controller_status.last_attempt.succeeded,
        old_status.controller_status.last_attempt.succeeded,
        ts
    )
    set_condition_timestamp(
        new_status.deploy_status.details.controller_status.last_attempt.succeeded,
        old_status.deploy_status.details.controller_status.last_attempt.succeeded,
        ts
    )


def make_status(rs, cur_state, ctl_error, use_deploy_status):
    """
    :type rs: yp.data_model.TReplicaSet
    :type cur_state: infra.rsc.src.model.state.CurrentState
    ;type ctl_error: Exception
    :type use_deploy_status: bool
    :rtype: yp.data_model.TReplicaSetStatus
    """
    rv = data_model.TReplicaSetStatus()

    rv.revision_id = rs.spec.revision_id
    target = int(rs.spec.revision_id)
    fill_deploy_status(rv, cur_state, target,
                       ctl_error,
                       replica_count=rs.spec.replica_count,
                       use_deploy_status=False)
    if use_deploy_status:
        fill_deploy_status(rv.deploy_status.details,
                           cur_state, target,
                           ctl_error,
                           replica_count=rs.spec.replica_count,
                           use_deploy_status=True)
        rv.deploy_status.pod_set_id = rs.meta.id

    if use_deploy_status:
        p = rv.deploy_status.details.current_revision_progress
        fill_status_ready_in_progress(rv,
                                      rs.spec.replica_count,
                                      rs.spec.deployment_strategy,
                                      p)
    else:
        fill_status_ready_in_progress(rv,
                                      rs.spec.replica_count,
                                      rs.spec.deployment_strategy,
                                      rv.current_revision_progress)
    set_status_timestamps(new_status=rv, old_status=rs.status)
    return rv
