from __future__ import unicode_literals
import logging

import gevent
import yt.common
from infra.swatlib.gevent import greenthread
from infra.swatlib.gevent import geventutil as gutil
from infra.rsc.src.model.consts import DEFAULT_OBJECT_SELECTORS


class PodSetGc(greenthread.GreenThread):

    def __init__(self, name, client, rs_storage, pod_set_storage, sleep_secs):
        """
        :type name: str
        :type client: infra.rsc.src.model.ypclient.YpClient
        :type rs_storage: infra.rsc.src.model.storage.IndexedStorage
        :type pod_set_storage: infra.rsc.src.model.storage.IndexedStorage
        :type sleep_secs: int
        """
        super(PodSetGc, self).__init__()
        self.name = name
        self.client = client
        self.rs_storage = rs_storage
        self.pod_set_storage = pod_set_storage
        self.sleep_secs = sleep_secs
        self.log = logging.getLogger(self.name)

    def check_pod_set(self, ps_id):
        """
        :type ps_id: str
        :rtype bool
        """
        rs = self.client.get_replica_set_ignore(ps_id,
                                                selectors=DEFAULT_OBJECT_SELECTORS)
        return rs is None

    def remove_pod_sets(self, ids):
        # Do not check pod_sets' labels here because these pod_sets were selected with proper match_filter
        """
        :type ids: set[str]
        :rtype: set[str]
        """
        removed = set()
        for ps_id in ids:
            try:
                if not self.check_pod_set(ps_id):
                    continue
                self.client.remove_pod_set(ps_id)
                removed.add(ps_id)
            except (yt.common.YtError, yt.common.YtResponseError) as e:
                self.log.exception("removing podset '%s' "
                                   "failed with error: %s", ps_id, e)
        return removed

    def process_gc(self):
        """
        :rtype: set[str]
        """
        rs_ids = set(self.rs_storage.list_ids())
        remove_candidates = set()

        for ps_id in gutil.gevent_idle_iter(self.pod_set_storage.list_ids()):
            if ps_id not in rs_ids:
                remove_candidates.add(ps_id)
        return self.remove_pod_sets(remove_candidates)

    def run(self):
        while True:
            try:
                removed = self.process_gc()
            except Exception as e:
                self.log.exception("gc failed with error: %s", e)
            else:
                self.log.info("removed podsets: %s", ",".join(removed))
            finally:
                gevent.sleep(self.sleep_secs)
