from __future__ import unicode_literals
import logging

import gevent
from infra.swatlib.gevent import greenthread
from infra.swatlib.gevent import geventutil as gutil
from infra.rsc.src.model import cluster_condition
from infra.rsc.src.model.consts import DEFAULT_OBJECT_SELECTORS


class Reflector(greenthread.GreenThread):

    def __init__(self, name, cluster,
                 obj_type, obj_class, obj_filter,
                 storage, client, batch_size, sleep_secs,
                 metrics_registry, runner_q=None):
        """
        :type name: str
        :type cluster: str
        :type obj_type: yp.data_model.EObjectType
        :type obj_filter: str
        :type storage: infra.rsc.src.model.storage.IndexedStorage
        :type client: infra.rsc.src.model.yp_client.YpClient
        :type batch_size: int
        :type sleep_secs: int
        :type metrics_registry: infra.swatlib.metrics.Registry
        :type runner_q: gevent.queue.Queue | None
        """
        super(Reflector, self).__init__()
        self.name = name
        self.cluster = cluster
        self.log = logging.getLogger('{}({})'.format(self.name, self.cluster))
        self.obj_type = obj_type
        self.obj_class = obj_class
        self.obj_filter = obj_filter
        self.storage = storage
        self.client = client
        self.batch_size = batch_size
        self.obj_filter = obj_filter
        self.sleep_secs = sleep_secs
        self.runner_q = runner_q
        self._metrics_registry = metrics_registry

    def sync(self):
        timestamp = self.client.generate_timestamp()
        g = self.client.select_all_objects(obj_type=self.obj_type,
                                           obj_class=self.obj_class,
                                           query=self.obj_filter,
                                           timestamp=timestamp,
                                           selectors=DEFAULT_OBJECT_SELECTORS,
                                           batch_size=self.batch_size)

        objs = []
        for obj in gutil.gevent_idle_iter(g):
            objs.append(obj)

        self.storage.replace(objs, timestamp)
        self.log.info("storage synced with %d objects: storage size: %d",
                      len(objs), len(self.storage.list_ids()))
        m = 'sync_timestamp_{}_{}'.format(self.name, self.cluster)
        g = self._metrics_registry.get_gauge(m, 'dhhh')
        g.set(int(timestamp))

    def put_to_runner_q(self, cc):
        """
        :type cc: infra.rsc.src.model.cluster_condition.ClusterCondition
        """
        if not self.runner_q:
            return

        # We need to clear queue and put new cluster_condition element.
        # Reflector is the only producer for queue so we can be sure
        # that clearing is safe.
        while True:
            try:
                self.runner_q.get_nowait()
            except gevent.queue.Empty:
                self.runner_q.put(cc)
                return

    def run(self):
        while True:
            try:
                self.sync()
            except Exception as e:
                self.log.exception("reflector sync failed with error: %s: "
                                   "sleeping for %d seconds...",
                                   e, self.sleep_secs)
                cc = cluster_condition.ClusterCondition(self.cluster)
                cc.set_exception(e)
                self.put_to_runner_q(cc)
            else:
                self.log.info("reflector synced: sleeping for %d seconds...",
                              self.sleep_secs)
                cc = cluster_condition.ClusterCondition(self.cluster)
                cc.set_ok()
                self.put_to_runner_q(cc)
            gevent.sleep(self.sleep_secs)
