from __future__ import unicode_literals
import itertools

import logging

from infra.swatlib.gevent import geventutil as gutil
from infra.rsc.src.lib import podutil
from infra.rsc.src.lib import rsutil
from infra.rsc.src.model import pod_id_generator
from infra.rsc.src.model import state
from infra.rsc.src.model import validation
from infra.rsc.src.model.consts import DEFAULT_OBJECT_SELECTORS


class ReplicaSetPodsUpdater(object):

    def __init__(self, client, match_labels,
                 is_inplace_update_disabled,
                 use_deploy_status,
                 pod_maker,
                 rate_limiter):
        """
        :type client: infra.rsc.src.model.ypclient.YpClient
        :type match_labels: yp.data_model.TAttributeDictionary
        :type is_inplace_update_disabled: bool
        :type use_deploy_status: bool
        """
        super(ReplicaSetPodsUpdater, self).__init__()
        self.client = client
        self._match_labels = match_labels
        self.is_inplace_update_disabled = is_inplace_update_disabled
        self.use_deploy_status = use_deploy_status
        self.pod_maker = pod_maker
        self.rate_limiter = rate_limiter

    @staticmethod
    def make_current_state(pod_lister):
        """
        :type pod_lister: infra.rsc.src.model.lister.PodLister
        :rtype: infra.rsc.src.model.state.CurrentState
        """
        s = state.CurrentState()
        for p in gutil.gevent_idle_iter(pod_lister.list_all()):
            s.add(p)
        return s

    def update_rs_status(self, rs, status):
        """
        :type rs: yp.data_model.TReplicaSet
        :type status: yp.data_model.TReplicaSetStatus
        """
        self.log.info('updating rs status...')
        try:
            self.client.update_replica_set_status(rs.meta.id, status)
        except Exception as e:
            self.log.exception("updating mcrs status failed with error: %s", e)

    def process_pods(self, rs, pod_lister, cur_state):
        """
        :type rs: yp.data_model.TReplicaSet
        :type pod_lister: infra.rsc.src.model.lister.PodLister
        :type cur_state: infra.mcrsc.src.model.state.CurrentState
        """
        target = int(rs.spec.revision_id)
        diff = cur_state.count() - rs.spec.replica_count

        if diff > 0:
            ids = itertools.islice(cur_state.find_pod_ids_to_process(target),
                                   diff)
            return self.remove_pods(pod_lister.list_by_ids(ids), rs.meta.id)
        elif diff < 0:
            return self.create_pods(rs, abs(diff), pod_lister=pod_lister)

        step = (rs.spec.deployment_strategy.max_unavailable -
                cur_state.in_progress.count(target))
        if step <= 0:
            return

        # Real replica_count is equal to spec's replica_count here.
        # At first we need to evict requested pods.
        to_evict = list(itertools.islice(cur_state.eviction_requested.all(), step))
        if to_evict:
            self.acknowledge_pods_eviction(pod_lister.list_by_ids(to_evict), rs.meta.id)
            return

        # Consider example: replica_count = 3, max_unavailable = 2, ready_actuals =
        # 2, ready_outdated = 1. In this case step = 2, but we need to update
        # only 1 pod, so we reduce step.
        limit = cur_state.count() - cur_state.ready.count(target)
        step = min(step, limit)
        if step <= 0:
            return

        if cur_state.count(target) == rs.spec.replica_count:
            return

        to_update = []
        to_replace = []
        ids = itertools.islice(cur_state.find_pod_ids_to_process(target),
                               step)
        template = self.pod_maker.make_pod_template(rs)
        for p in pod_lister.list_by_ids(ids):
            can_update = (
                not self.is_inplace_update_disabled and
                podutil.is_pod_spec_updateable(p.spec, template.spec)
            )
            if can_update:
                to_update.append(p)
            else:
                to_replace.append(p)

        if to_replace:
            self.replace_pods(to_replace, rs, pod_lister=pod_lister)
        if to_update:
            self.update_pods(to_update, template=template)

    def create_pods(self, rs, n, pod_lister):
        self.log.info('creating %s pods', n)
        g = pod_id_generator.make_pod_id_generator(rs, pod_lister)
        pods = self.pod_maker.make_pods(rs, n, pod_id_generator=g)
        tid, ts = self.client.start_transaction()
        ps = self.client.get_pod_set_ignore(rs_id=rs.meta.id,
                                            timestamp=ts,
                                            selectors=DEFAULT_OBJECT_SELECTORS)
        if ps:
            validation.validate_pod_set_labels(ps, self._match_labels)
        else:
            ps = self.pod_maker.make_pod_set(rs)
            self.client.create_pod_set(ps=ps, transaction_id=tid)
        self.client.create_pods(pods=pods, transaction_id=tid)
        self.client.commit_transaction(tid)
        self.rate_limiter.update_last_process_time(rs.meta.id)

    def remove_pods(self, pods, rs_id):
        if not self.rate_limiter.is_process_allowed(rs_id):
            return
        self.log.info('removing %s pods', len(pods))
        self.client.remove_pods(pods)
        self.rate_limiter.update_last_process_time(rs_id)

    def replace_pods(self, pods, rs, pod_lister):
        if not self.rate_limiter.is_process_allowed(rs.meta.id):
            return
        self.log.info('replacing %s pods', len(pods))
        g = pod_id_generator.make_pod_id_generator(rs, pod_lister)
        new_pods = self.pod_maker.make_pods(rs, len(pods), pod_id_generator=g)
        tid, ts = self.client.start_transaction()
        self.client.remove_pods(pods, tid)
        self.client.create_pods(pods=new_pods, transaction_id=tid)
        self.client.commit_transaction(tid)
        self.rate_limiter.update_last_process_time(rs.meta.id)

    def update_pods(self, pods, template):
        self.log.info('updating %s pods', len(pods))
        self.client.update_pods(pods=pods, template=template)

    def update_pod_set(self, rs):
        ps = self.client.get_pod_set_ignore(rs.meta.id,
                                            timestamp=None,
                                            selectors=DEFAULT_OBJECT_SELECTORS)
        if not ps:
            return

        validation.validate_pod_set_labels(ps, self._match_labels)
        template = self.pod_maker.make_pod_set(rs)

        # NOTE: we do not allow to update node_segment_id, because YP cannot
        # handle it properly.
        template.spec.node_segment_id = ps.spec.node_segment_id

        need_update = podutil.update_pod_set_needed(ps, template)
        if not need_update:
            return

        self.log.info('updating pod_set')
        self.client.update_pod_set(ps_id=ps.meta.id, template=template)

    def acknowledge_pods_eviction(self, pods, rs_id):
        if not self.rate_limiter.is_process_allowed(rs_id):
            return
        self.log.info('evicting %s pods', len(pods))
        msg = "Acknowledged by RSC"
        self.client.update_pods_acknowledge_eviction(pods, msg)
        self.rate_limiter.update_last_process_time(rs_id)

    def update_allocations(self, rs, pod_lister):
        """
        :type rs: yp.data_model.TReplicaSet
        :type pod_lister: infra.rsc.src.model.lister.PodLister
        """
        self.log = logging.getLogger('rs({})'.format(rs.meta.id))

        cur_state = self.make_current_state(pod_lister)
        self.log.info('pods: %s', podutil.stringify_revisions(rs.status.revisions_progress))

        # Copy rs revision to pods.
        rs.spec.pod_template_spec.spec.pod_agent_payload.spec.revision = int(rs.spec.revision_id)

        self.log.info('updating pod allocations')
        ctl_error = None
        try:
            self.update_pod_set(rs)
            self.process_pods(rs, pod_lister, cur_state)
        except Exception as e:
            self.log.exception("updating pod allocations failed with error: %s", e)
            ctl_error = e
        status = rsutil.make_status(rs, cur_state, ctl_error,
                                    use_deploy_status=self.use_deploy_status)
        if status != rs.status:
            self.update_rs_status(rs, status)
