from __future__ import unicode_literals
import logging

import gevent
from infra.swatlib.gevent import greenthread
from infra.swatlib.gevent import geventutil as gutil
from infra.rsc.src.model import lister


class CtlRunner(greenthread.GreenThread):

    def __init__(self, ctl, q, cluster,
                 rs_storage, pod_storage,
                 threads_count,
                 metrics_registry):
        """
        :type ctl: infra.rsc.src.model.rs_updater.ReplicaSetPodsUpdater
        :type q: gevent.queue.Queue
        :type cluster: str
        :type rs_storage: infra.rsc.src.model.storage.IndexedStorage
        :type pod_storage: infra.rsc.src.model.storage.PodIndexedStorage
        :type threads_count: int
        :type metrics_registry: infra.swatlib.metrics.Registry
        """
        super(CtlRunner, self).__init__()
        self.ctl = ctl
        self.q = q
        self.cluster = cluster
        self.rs_storage = rs_storage
        self.pod_storage = pod_storage
        self.log = logging.getLogger('ctl-runner')
        self.last_run_time = None
        self._metrics_registry = metrics_registry
        self._gs = []  # List of active controller greenlets
        self._pool = gevent.pool.Pool(threads_count)

    def _handle_ctl_exception(self, exc, rs_id):
        """
        :type exc: Exception
        :type rs_id: str
        """
        self.log.exception("controller for rs '%s' finished with error: %s",
                           rs_id, exc)

    def start_ctl(self, rs_id):
        """
        :type rs_id: str
        """
        self.log.info("starting controller for rs '%s'...", rs_id)
        rs = self.rs_storage.get(rs_id)
        pod_lister = lister.PodLister(rs_id, self.pod_storage)
        g = gevent.Greenlet(self.ctl.update_allocations, rs, pod_lister)
        g.link_exception(lambda e: self._handle_ctl_exception(e, rs_id))
        self._gs.append(g)
        self._pool.start(g)
        self.log.info("started controller for rs '%s'", rs_id)

    def run(self):
        try:
            while True:
                self.log.info("waiting for next sync event...")
                cc = self.q.get()
                if cc.status != "OK":
                    self.log.info("cluster condition is not ok: %s: "
                                  "controllers will not be started",
                                  cc)
                    continue

                m = 'iteration_time_{}_rsc'.format(self.cluster)
                t = self._metrics_registry.get_histogram(m).timer()
                for rs_id in self.rs_storage.list_ids():
                    self.start_ctl(rs_id)
                gevent.wait(self._gs)
                del self._gs[:]
                t.stop()
                self.log.info("iteration_time %s",
                              [[b, v] for b, v in t.hgram._buckets if v != 0])
        finally:
            self.log.info("stopping all controllers...")
            for g in self._gs:
                gutil.force_kill_greenlet(g, ignore_greenlet_exit=True,
                                          log=self.log)
            del self._gs[:]
            self.log.info("stopped all controllers")
