from __future__ import unicode_literals
import collections
import itertools

from infra.rsc.src.lib import podutil


class RevisionIndex(object):

    def __init__(self):
        self.idx = collections.defaultdict(set)

    def __iter__(self):
        return self.idx.iterkeys()

    def __nonzero__(self):
        return bool(self.idx)

    def add(self, pod):
        """
        :type pod: yp.data_model.TPod
        """
        r = pod.spec.pod_agent_payload.spec.revision
        self.idx[r].add(pod.meta.id)

    def count(self, revision=None):
        """
        :type revision: int | None
        :rtype: int
        """
        if revision is not None:
            return len(self.idx[revision])
        return sum(len(ids) for ids in self.idx.itervalues())

    def all(self):
        """
        :rtype: collections.Iterable[str]
        """
        return itertools.chain.from_iterable(self.idx.itervalues())

    def find_by_revision(self, revision):
        """
        :type revision: int
        :rtype: collections.Iterable[str]
        """
        return iter(self.idx[revision])

    def exclude_revision(self, revision):
        """
        :type revision: int
        :rtype: collections.Iterable[str]
        """
        its = []
        sorted_revisions = sorted(self, reverse=True)
        for r in sorted_revisions:
            if r == revision:
                continue
            its.append(self.find_by_revision(r))
        return itertools.chain.from_iterable(its)


class CurrentState(object):

    def __init__(self):
        self.in_progress = RevisionIndex()
        self.ready = RevisionIndex()
        self.eviction_requested = RevisionIndex()

    def add(self, pod):
        """
        :type pod: yp.data_model.TPod
        """
        is_ready = podutil.is_pod_alive(pod)
        if is_ready:
            self.ready.add(pod)
        else:
            self.in_progress.add(pod)

        if podutil.is_pod_finally_dead(pod):
            self.eviction_requested.add(pod)

    def find_pod_ids_to_process(self, target):
        """
        :type target: int
        :rtype: collections.Iterable[str]
        """
        for p_id in self.in_progress.exclude_revision(target):
            yield p_id
        for p_id in self.ready.exclude_revision(target):
            yield p_id
        for p_id in self.in_progress.find_by_revision(target):
            yield p_id
        for p_id in self.ready.find_by_revision(target):
            yield p_id

    def count(self, revision=None):
        """
        :type revision: int | None
        :rtype: int
        """
        return self.ready.count(revision) + self.in_progress.count(revision)
