from __future__ import unicode_literals
import collections

from infra.swatlib.gevent import geventutil as gutil
from infra.rsc.src.lib import podutil


class IndexedStorage(object):
    def __init__(self):
        self.index = {}
        self.last_replace_time = None

    def get_obj_id(self, obj):
        return obj.meta.id

    def get(self, obj_id):
        return self.index.get(obj_id)

    def list_ids(self):
        return self.index.keys()

    def put(self, obj):
        obj_id = self.get_obj_id(obj)
        self.index[obj_id] = obj

    def merge(self, other):
        self.index = other.index

    def replace(self, objs, timestamp):
        new_storage = self.make_from_objs(objs)
        self.merge(new_storage)
        self.last_replace_time = timestamp

    @classmethod
    def make_from_objs(cls, objs):
        new_storage = cls()
        for obj in gutil.gevent_idle_iter(objs):
            new_storage.put(obj)
        return new_storage


class PodIndexedStorage(IndexedStorage):

    def __init__(self):
        self.index = {}
        super(PodIndexedStorage, self).__init__()
        self.ps_index = collections.defaultdict(list)
        self.stage_index = collections.defaultdict(list)

    def list_by_ps_id(self, ps_id):
        """
        :type ps_id: str
        :rtype: list[yp.data_model.TPod]
        """
        return self.ps_index.get(ps_id) or []

    def list_by_stage_id(self, stage_id):
        """
        :type stage_id: str
        :rtype: list[yp.data_model.TPod]
        """
        return self.stage_index.get(stage_id) or []

    def put(self, obj):
        """
        :type obj: yp.data_model.TPod
        """
        super(PodIndexedStorage, self).put(obj)
        self.ps_index[obj.meta.pod_set_id].append(obj)
        stage_id = podutil.get_stage_id_from_pod_id(obj.meta.id)
        if stage_id:
            self.stage_index[stage_id].append(obj)

    def merge(self, other):
        """
        :type other: infra.rsc.src.model.storage.pod.PodIndexedStorage
        """
        super(PodIndexedStorage, self).merge(other)
        self.ps_index = other.ps_index
        self.stage_index = other.stage_index
