from __future__ import unicode_literals

import yp.client
import yp.common
import yt.yson as yson
import yt_yson_bindings
from gevent import threadpool
from infra.swatlib.gevent import geventutil
from infra.rsc.src.model.consts import DEFAULT_OBJECT_SELECTORS
import yp.data_model as data_model
from yp_proto.yp.client.api.proto import object_service_pb2


class MaxIdQueryMaker(object):

    __slots__ = ('base_query', '_id')

    def __init__(self, base_query):
        self.base_query = base_query
        self._id = None

    def update(self, obj):
        if not self._id or self._id < obj.meta.id:
            self._id = obj.meta.id

    def make_query(self):
        if not self._id:
            return self.base_query
        return '({}) AND [/meta/id] > "{}"'.format(self.base_query, self._id)


class PodMaxIdQueryMaker(object):

    __slots__ = ('base_query', '_id_tuple')

    ID_TEMPLATE = '({}) AND ([/meta/pod_set_id], [/meta/id]) > ("{}", "{}")'

    def __init__(self, base_query):
        self.base_query = base_query
        self._id_tuple = None

    def update(self, obj):
        id_tuple = obj.meta.pod_set_id, obj.meta.id
        if not self._id_tuple or self._id_tuple < id_tuple:
            self._id_tuple = id_tuple

    def make_query(self):
        if not self._id_tuple:
            return self.base_query
        return self.ID_TEMPLATE.format(self.base_query, *self._id_tuple)


class YpClient(object):

    def __init__(self, stub):
        self.stub = stub
        self.tp = threadpool.ThreadPool(maxsize=10)

    def _get_object_values(self, object_type, object_id,
                           selectors=None, timestamp=None, ignore_nonexistent=False):
        """
        :type object_type: str
        :type object_id: str
        :type selectors: str | None
        :type timestamp: str | None
        :rtype: list[str]
        """
        req = object_service_pb2.TReqGetObject()
        req.object_type = object_type
        req.object_id = object_id
        if timestamp is not None:
            req.timestamp = timestamp
        selectors = selectors or DEFAULT_OBJECT_SELECTORS
        req.selector.paths.extend(selectors)
        if ignore_nonexistent:
            req.options.ignore_nonexistent = True
        return self.tp.apply(self.stub.GetObject, [req]).result.values

    def _select_objects_values(self, object_type, limit,
                               query=None, selectors=None, timestamp=None):
        """
        :type object_type: str
        :type limit: int
        :type timestamp: int | None
        :type query: str | None
        :rtype: generator[list[bytes]]
        """
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = object_type
        req.limit.value = limit
        if timestamp is not None:
            req.timestamp = timestamp
        selectors = selectors or ['']
        req.selector.paths.extend(selectors)
        if query:
            req.filter.query = query
        req.format = object_service_pb2.PF_PROTOBUF
        req.options.fetch_root_object = True

        resp = self.tp.apply(self.stub.SelectObjects, [req])
        for r in resp.results:
            yield r.value_payloads

    def _object_exists(self, object_type, object_id, timestamp=None):
        """
        :type object_type: str
        :type object_id: str
        :type timestamp: int | None
        :rtype: bool
        """
        v = self._get_object_values(object_id=object_id,
                                    object_type=object_type,
                                    timestamp=timestamp,
                                    ignore_nonexistent=True)
        return bool(v)

    def _create_object(self, object_pb, object_type, transaction_id=None):
        """
        :type object_pb: yp.data_model.T
        :type object_type: str
        :type transaction_id: str | None
        :rtype: str
        """
        req = object_service_pb2.TReqCreateObject()

        if transaction_id:
            req.transaction_id = transaction_id

        req.object_type = object_type
        req.attributes = yt_yson_bindings.dumps_proto(object_pb)
        rsp = self.tp.apply(self.stub.CreateObject, [req])
        return rsp.object_id

    def _create_objects(self, objects_pb, object_type, transaction_id=None):
        """
        :type objects_pb: list[yp.data_model.T]
        :type object_type: str
        :type transaction_id: str | None
        :rtype: list[str]
        """
        req = object_service_pb2.TReqCreateObjects()

        if transaction_id:
            req.transaction_id = transaction_id

        for o in objects_pb:
            subreq = req.subrequests.add()
            subreq.object_type = object_type
            subreq.attributes = yt_yson_bindings.dumps_proto(o)

        rsp = self.tp.apply(self.stub.CreateObjects, [req])
        return [r.object_id for r in rsp.subresponses]

    def generate_timestamp(self):
        """
        :rtype: int
        """
        req = object_service_pb2.TReqGenerateTimestamp()
        resp = self.tp.apply(self.stub.GenerateTimestamp, [req])
        return resp.timestamp

    def start_transaction(self):
        """
        :rtype: (unicode, int)
        """
        req = object_service_pb2.TReqStartTransaction()
        resp = self.tp.apply(self.stub.StartTransaction, [req])
        return resp.transaction_id, resp.start_timestamp

    def commit_transaction(self, transaction_id):
        """
        :type transaction_id: unicode
        """
        req = object_service_pb2.TReqCommitTransaction()
        req.transaction_id = transaction_id
        self.tp.apply(self.stub.CommitTransaction, [req])

    def get_pod_set_ignore(self, rs_id, timestamp=None, selectors=None):
        """
        :type rs_id: str
        :type timestamp: int
        :rtype: yp.data_model.TPodSet
        """
        v = self._get_object_values(object_id=rs_id,
                                    object_type=data_model.OT_POD_SET,
                                    timestamp=timestamp,
                                    selectors=selectors,
                                    ignore_nonexistent=True)
        if not v:
            return None
        return self._load_object(data_model.TPodSet, selectors, v)

    def get_replica_set_ignore(self, rs_id, selectors=None):
        """
        :type rs_id: str
        :rtype: yp.data_model.TReplicaSet
        """
        v = self._get_object_values(object_id=rs_id,
                                    object_type=data_model.OT_REPLICA_SET,
                                    selectors=selectors,
                                    ignore_nonexistent=True)
        if not v:
            return None
        return self._load_object(data_model.TReplicaSet, selectors, v)

    def get_multi_cluster_replica_set_ignore(self, mcrs_id, selectors=None):
        """
        :type mcrs_id: str
        :rtype: yp.data_model.TMultiClusterReplicaSet
        """
        v = self._get_object_values(object_id=mcrs_id,
                                    object_type=data_model.OT_MULTI_CLUSTER_REPLICA_SET,
                                    selectors=selectors,
                                    ignore_nonexistent=True)
        if not v:
            return None
        return self._load_object(data_model.TMultiClusterReplicaSet, selectors, v)

    def pod_set_exists(self, ps_id, timestamp=None):
        """
        :type ps_id: str
        :type timestamp: int | None
        :rtype: bool
        """
        return self._object_exists(object_type=data_model.OT_POD_SET,
                                   object_id=ps_id,
                                   timestamp=timestamp)

    def get_replica_set(self, rs_id, timestamp, selectors):
        """
        :type rs_id: str
        :type timestamp: int
        :rtype: yp.data_model.TReplicaSet
        """
        vals = self._get_object_values(object_id=rs_id,
                                       object_type=data_model.OT_REPLICA_SET,
                                       timestamp=timestamp,
                                       selectors=selectors)
        return self._load_object(data_model.TReplicaSet, selectors, vals)

    def select_all_objects(self, obj_type, obj_class, batch_size,
                           query=None, timestamp=None, selectors=None):
        if selectors is None:
            selectors = [""]

        if obj_type == data_model.OT_POD:
            query_maker = PodMaxIdQueryMaker(query)
        else:
            query_maker = MaxIdQueryMaker(query)

        while True:
            q = query_maker.make_query()
            objs_values = self._select_objects_values(object_type=obj_type,
                                                      limit=batch_size,
                                                      query=q,
                                                      timestamp=timestamp,
                                                      selectors=selectors)
            count = 0
            for v in objs_values:
                obj = obj_class()
                obj.MergeFromString(v[0].protobuf)
                yield obj
                count += 1
                query_maker.update(obj)
            if count < batch_size:
                return

    def create_pod_set(self, ps, transaction_id):
        """
        :type ps: yp.data_model.TPodSet
        :type transaction_id: str
        :rtype: str
        """
        return self._create_object(object_pb=ps,
                                   object_type=data_model.OT_POD_SET,
                                   transaction_id=transaction_id)

    def create_pods(self, pods, transaction_id):
        """
        :type pods: list[yp.data_model.TPod]
        :type transaction_id: str
        :rtype list[str]
        """
        return self._create_objects(objects_pb=pods,
                                    object_type=data_model.OT_POD,
                                    transaction_id=transaction_id)

    def update_pod_set(self, ps_id, template):
        """
        :type ps_id: str
        :type template: yp.data_model.TPodSet
        """
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = data_model.OT_POD_SET
        req.object_id = ps_id

        upd = req.set_updates.add()
        upd.path = '/spec'
        upd.value = yt_yson_bindings.dumps_proto(template.spec)

        upd = req.set_updates.add()
        upd.path = '/meta/acl'
        acl = []
        for entry in template.meta.acl:
            acl.append(yp.common.protobuf_to_dict(entry))
        upd.value = yson.dumps(acl)

        self.tp.apply(self.stub.UpdateObject, [req])

    def update_pods(self, pods, template):
        """
        :type pods: collections.Iterable[yp.data_model.TPod]
        :type pod_agent_payload: yp_proto.yp.client.api.proto.data_model_pb2.TPodSpec
        """
        req = object_service_pb2.TReqUpdateObjects()
        for p in geventutil.gevent_idle_iter(pods):
            template.spec.pod_agent_payload.spec.id = p.meta.id
            for path, field in [('/spec', template.spec), ('/labels', template.labels)]:
                subreq = req.subrequests.add()
                subreq.object_type = data_model.OT_POD
                subreq.object_id = p.meta.id
                upd = subreq.set_updates.add()
                upd.path = path
                upd.value = yt_yson_bindings.dumps_proto(field)
        self.tp.apply(self.stub.UpdateObjects, [req])

    def update_pods_acknowledge_eviction(self, pods, msg):
        """
        :type pods: list[yp.data_model.TPod]
        :type msg: str
        """
        req = object_service_pb2.TReqUpdateObjects()
        yson_value = yson.dumps({'message': msg})
        for p in geventutil.gevent_idle_iter(pods):
            subreq = req.subrequests.add()
            subreq.object_type = data_model.OT_POD
            subreq.object_id = p.meta.id
            upd = subreq.set_updates.add()
            upd.path = '/control/acknowledge_eviction'
            upd.value = yson_value
        self.tp.apply(self.stub.UpdateObjects, [req])

    def update_replica_set_status(self, rs_id, status):
        """
        :type rs_id: str
        :type status: yp.data_model.TReplicaSetStatus
        """
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = data_model.OT_REPLICA_SET
        req.object_id = rs_id
        upd = req.set_updates.add()
        upd.path = '/status'
        upd.value = yt_yson_bindings.dumps_proto(status)
        self.tp.apply(self.stub.UpdateObject, [req])

    def update_replica_set_annotations(self, rs_id, annotations, transaction_id):
        """
        :type rs_id: str
        :type annotations: yp.data_model.TAttributeDictionary
        """
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = data_model.OT_REPLICA_SET
        req.object_id = rs_id
        req.transaction_id = transaction_id

        upd = req.set_updates.add()
        upd.path = '/annotations'
        upd.value = yt_yson_bindings.dumps_proto(annotations)
        self.tp.apply(self.stub.UpdateObject, [req])

    def update_multi_cluster_replica_set_status(self, mcrs_id, status):
        """
        :type mcrs_id: str
        :type status: yp.data_model.TMultiClusterReplicaSetStatus
        """
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = data_model.OT_MULTI_CLUSTER_REPLICA_SET
        req.object_id = mcrs_id
        upd = req.set_updates.add()
        upd.path = '/status'
        upd.value = yt_yson_bindings.dumps_proto(status)
        self.tp.apply(self.stub.UpdateObject, [req])

    def remove_pods(self, pods, transaction_id=None):
        """
        :type pods: collections.Iterable[yp.data_model.TPod]
        :type transaction_id: str | None
        """
        req = object_service_pb2.TReqRemoveObjects()
        if transaction_id:
            req.transaction_id = transaction_id
        for p in geventutil.gevent_idle_iter(pods):
            sub = req.subrequests.add()
            sub.object_type = data_model.OT_POD
            sub.object_id = p.meta.id
        self.tp.apply(self.stub.RemoveObjects, [req])

    def remove_pod_set(self, pod_set_id):
        """
        :type pod_set_id: str
        """
        req = object_service_pb2.TReqRemoveObject()
        req.object_type = data_model.OT_POD_SET
        req.object_id = pod_set_id
        self.tp.apply(self.stub.RemoveObject, [req])

    @staticmethod
    def _make_attr_paths(selectors):
        return tuple(s.strip("/").split("/") for s in selectors)

    def _load_object(self, obj_class, selectors, values):
        paths = self._make_attr_paths(selectors)
        return self._load_object_attrs(obj_class, paths, values)

    @staticmethod
    def _make_object_attr(obj, attr_path):
        attr = obj
        for attr_name in attr_path:
            if attr_name:
                attr = getattr(attr, attr_name)
        return attr

    def _load_object_attrs(self, obj_class, paths, values):
        obj = obj_class()
        for i in xrange(len(paths)):
            attr = self._make_object_attr(obj, paths[i])
            o = yt_yson_bindings.loads_proto(values[i], type(attr),
                                             skip_unknown_fields=True)
            attr.CopyFrom(o)
        return obj
