package cmd

import (
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"io/ioutil"
	"net/http"
	"os"
	"path/filepath"
	"regexp"
	"sync"
	"time"
)

var (
	url          string
	pth          string
	host         string
	date         string
	ptrnFileName string
	dl           string
	workersCount int

	uploadCmd = &cobra.Command{
		Use:   "upload",
		Short: "Upload Oops to coroner server",
		Run:   func(cmd *cobra.Command, args []string) { runUpload() },
	}
)

type session struct {
	ts   int64
	host string
	fp   string
}

func init() {
	uploadCmd.Flags().StringVar(&url, "u", "http://localhost:80/api/v1/new_oops", "coroner url")
	uploadCmd.Flags().StringVar(&pth, "p", "./oops/*", "regexp to files")
	uploadCmd.Flags().StringVar(&host, "h", "localhost", "hostname")
	uploadCmd.Flags().StringVar(&date, "d", time.Now().Format("2006-01-02"), "date")
	uploadCmd.Flags().StringVar(&ptrnFileName, "ptrn", "(?P<host>[a-z0-9-.]+)\\-(?P<date>[0-9]{4}\\-[0-9]{2}\\-[0-9]{2})", "ptrn for getting 'host' and 'date' from filename")
	uploadCmd.Flags().StringVar(&dl, "dl", "2006-01-02", "date layout for parse date, see https://golang.org/pkg/time/")
	uploadCmd.Flags().IntVar(&workersCount, "w", 10, "workers count")
	rootCmd.AddCommand(uploadCmd)
}

func runUpload() {
	var wg sync.WaitGroup

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	rgxpFileName := regexp.MustCompile(ptrnFileName)

	files, err := filepath.Glob(pth)
	if err != nil {
		panic(err)
	}

	sessionQueue := make(chan *session, workersCount)
	for id := 1; id <= workersCount; id++ {
		wg.Add(1)
		go uploadWorker(ctx, &wg, id, sessionQueue)
	}

	fmt.Printf("found files: %s\n", files)
	for _, f := range files {
		s := &session{fp: f}
		fmt.Printf("getting info from %s\n", f)
		opts := parseString(rgxpFileName, filepath.Base(f))
		if h, ok := opts["host"]; ok {
			s.host = h
		}
		if d, ok := opts["date"]; ok {
			date = d
		}
		t, err := time.Parse(dl, date)
		if err != nil {
			fmt.Println(err)
			continue
		}
		s.ts = t.UnixMicro()

		sessionQueue <- s
	}

	cancel()
	wg.Wait()
}

func uploadWorker(ctx context.Context, wg *sync.WaitGroup, id int, queue chan *session) {
	fmt.Printf("starting upload worker #%d\n", id)
	defer wg.Done()

	c := &http.Client{}

	for {
		select {
		case <-ctx.Done():
			fmt.Printf("%d: exit by ctx\n", id)
			return
		case s := <-queue:
			data, err := upload(c, s)
			if err != nil {
				fmt.Printf("%d: %s\n", id, err)
				continue
			}
			fmt.Printf("%d: uploaded %s, %s\n", id, s.fp, data)
		}
	}
}

func upload(c *http.Client, s *session) ([]byte, error) {
	f, err := os.Open(s.fp)
	if err != nil {
		return nil, err
	}
	req, err := http.NewRequest("POST", url, f)
	if err != nil {
		return nil, err
	}

	q := req.URL.Query()
	q.Add("ts", fmt.Sprintf("%d", s.ts))
	q.Add("host", s.host)
	req.URL.RawQuery = q.Encode()

	resp, err := c.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()

	return ioutil.ReadAll(resp.Body)
}

func parseString(rgxp *regexp.Regexp, s string) map[string]string {
	keys := rgxp.SubexpNames()
	values := rgxp.FindStringSubmatch(s)
	d := make(map[string]string)
	for i := 1; i < len(values); i++ {
		if values[i] != "" {
			d[keys[i]] = values[i]
		}
	}
	return d
}
