package coroner

import (
	"context"
	"encoding/json"
	"os"
	"sync"
	"syscall"
	"time"

	"a.yandex-team.ru/infra/rsm/coroner/internal"
	log "a.yandex-team.ru/infra/rsm/coroner/internal/logger"
	"a.yandex-team.ru/infra/rsm/dnsmanager/pkg/pushyasm"
	"a.yandex-team.ru/yt/go/guid"
)

var (
	wg               sync.WaitGroup
	oopsParsersCount = 10
	oopsQueueSize    = 1000 * 1024
	sessionQueueSize = 10
)

type Session struct {
	TS   int64
	Addr string
	Host string
	Data []byte
}

type RawData struct {
	TS   int64
	Host string
	Data string
	ID   string
}

//for unification data given to Logbroker
func (rd *RawData) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"ts":   rd.TS,
		"host": rd.Host,
		"data": rd.Data,
		"id":   rd.ID,
	})
}

func NewRD(s *Session) *RawData {
	return &RawData{
		TS:   s.TS,
		Host: s.Host,
		Data: string(s.Data[:]),
		ID:   guid.New().String(),
	}
}

func (ses *Session) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"ts": ses.TS,
		//"Addr": ses.Addr,
		"host": ses.Host,
		"data": string(ses.Data[:]),
	})
}

func Run(c *Config) {
	ctx, cancel := internal.WithCancelOnSignal(context.Background(), []os.Signal{syscall.SIGTERM, syscall.SIGINT, syscall.SIGKILL})
	defer cancel()

	sessionQueue := make(chan *Session, sessionQueueSize)
	oopsQueue := make(chan json.Marshaler, oopsQueueSize)
	rawQueue := make(chan json.Marshaler, sessionQueueSize)

	py := pushyasm.NewPush(c.YasmHost, c.YasmPort, map[string]string{
		"itype": c.YasmItype,
		"ctype": c.YasmCtype,
		"prj":   c.YasmPrj,
		"geo":   c.YasmGeo,
	})
	go py.Run(ctx)

	udpSrv, err := NewUDPSrv(c.UDPPort, sessionQueue, log.L)
	if err != nil {
		panic(err)
	}
	wg.Add(1)
	go udpSrv.Run(ctx, &wg)

	httpSrv, err := NewHTTPSrv(c.HTTPPort, sessionQueue, log.L)
	if err != nil {
		panic(err)
	}
	wg.Add(1)
	go httpSrv.Run(ctx, &wg)

	st := NewLBStorage(log.L, oopsQueue, rawQueue, c.LBDir)
	wg.Add(1)
	go st.Run(ctx, &wg)

	for id := 1; id <= oopsParsersCount; id++ {
		wg.Add(1)
		go RunOopsParser(ctx, &wg, log.L, id, sessionQueue, oopsQueue, rawQueue)
	}

	wg.Add(1)
	go func() {
		defer wg.Done()
		metricSessionQueue := pushyasm.NewNumeric("coroner-in_queue", pushyasm.Counter, pushyasm.Last)
		metricOopsQueue := pushyasm.NewNumeric("coroner-out_queue", pushyasm.Counter, pushyasm.Last)
		metricRawQueue := pushyasm.NewNumeric("coroner-raw-out_queue", pushyasm.Counter, pushyasm.Last)
		tt := time.NewTicker(5 * time.Second)

		for {
			select {
			case <-ctx.Done():
				return
			case <-tt.C:
				metricSessionQueue.Set(float64(len(sessionQueue)))
				metricOopsQueue.Set(float64(len(oopsQueue)))
				metricRawQueue.Set(float64(len(rawQueue)))
			}
		}
	}()

	wg.Wait()
}
