package coroner

import (
	"context"
	"encoding/json"
	"regexp"
	"strings"
	"sync"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/yt/go/guid"
)

var (
	skipMsgPtrns = []string{
		".*pid [0-9]+ .*, uid [0-9]+.*exited on signal.*",
	}
	skipMsgRgxp = regexp.MustCompile(strings.Join(skipMsgPtrns, "|"))
	// Oct 27 03:03:49 sas5-1555 kernel: [5014915.781919] IPv6: ADDRCONF(NETDEV_UP): L3-4271: link is not ready
	// trim to
	// IPv6: ADDRCONF(NETDEV_UP): L3-4271: link is not ready
	trimRgxp = regexp.MustCompile(`.*\[[ 0-9.]+\] (.*)`)
)

type Oops struct {
	ID         string              `yson:"id"`
	TS         int64               `yson:"ts"`
	Host       string              `yson:"host"`
	Severity   Severity            `yson:"severity"`
	Name       string              `yson:"name"`
	Message    string              `yson:"message"`
	Kernel     string              `yson:"kernel,omitempty"`
	Command    string              `yson:"command,omitempty"`
	Pid        string              `yson:"pid,omitempty"`
	CPU        string              `yson:"cpu,omitempty"`
	Dev        string              `yson:"dev,omitempty"`
	Extra      map[string][]string `yson:"extra,omitempty"`
	StackTrace []string            `yson:"stacktrace,omitempty"`
	// TODO: StackTraceStr is temporary field, should use yson in query
	StackTraceStr string   `yson:"stacktraceStr,omitempty"`
	Rip           string   `yson:"rip,omitempty"`
	WorkQueue     string   `yson:"workqueue,omitempty"`
	HWName        string   `yson:"hwname,omitempty"`
	LogRaw        []string `yson:"lograw,omitempty"`
}

func (o *Oops) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"id":            o.ID,
		"ts":            o.TS,
		"host":          o.Host,
		"severity":      o.Severity,
		"name":          o.Name,
		"message":       o.Message,
		"kernel":        o.Kernel,
		"command":       o.Command,
		"pid":           o.Pid,
		"cpu":           o.CPU,
		"dev":           o.Dev,
		"extra":         o.Extra,
		"stacktrace":    o.StackTrace,
		"stacktracestr": o.StackTraceStr,
		"rip":           o.Rip,
		"workqueue":     o.WorkQueue,
		"hwname":        o.HWName,
		"lograw":        o.LogRaw,
	})
}

func NewOops(ts int64, name string, severity Severity, addr string) *Oops {
	return &Oops{
		ID:       guid.New().String(),
		TS:       ts,
		Host:     addr,
		Severity: severity,
		Name:     name,
		LogRaw:   []string{},
		Extra:    map[string][]string{},
	}
}

func (o *Oops) AddLogRaw(s string) {
	o.LogRaw = append(o.LogRaw, s)
}

func (o *Oops) Parse(d map[string]string) {
	for k, v := range d {
		switch k {
		case "cpu":
			o.CPU = v
		case "dev":
			o.Dev = v
		case "pid":
			o.Pid = v
		case "comm":
			o.Command = v
		case "kernel":
			o.Kernel = v
		case "msg":
			o.Message = v
		case "stacktrace":
			o.StackTrace = append(o.StackTrace, strings.ReplaceAll(v, " ", ""))
			o.StackTraceStr = strings.Join(o.StackTrace, ";")
		case "rip":
			if o.Rip == "" {
				o.Rip = v
			}
		case "workqueue":
			o.WorkQueue = v
		case "hwname":
			o.HWName = v
		default:
			if _, ok := o.Extra[k]; !ok {
				o.Extra[k] = []string{}
			}
			o.Extra[k] = append(o.Extra[k], v)
		}
	}
}

func RunOopsParser(ctx context.Context, wg *sync.WaitGroup, l *zap.Logger, id int, sessionQueue chan *Session, oopsQueue chan json.Marshaler, rawQueue chan json.Marshaler) {
	l.Infof("starting OopsParser %d\n", id)
	defer wg.Done()

	for {
		select {
		case <-ctx.Done():
			l.Infof("OopsParser[%d] exit by ctx\n", id)
			return
		case s := <-sessionQueue:
			rawQueue <- NewRD(s)
			for _, o := range OopsParse(s) {
				//log.Debug.Printf("%d: push to oops queue: %#v\n", id, o)
				oopsQueue <- o
			}
		}
	}
}

func OopsParse(s *Session) []*Oops {
	var (
		o  *Oops
		ol []*Oops
		e  *Event
		ce *Event
	)
	for _, msg := range strings.Split(string(s.Data), "\n") {
		// trim message
		trimedMsg := trimRgxp.FindStringSubmatch(msg)
		if len(trimedMsg) == 0 {
			continue
		}
		msg = trimedMsg[1]

		// skip info messages
		if skipMsgRgxp.MatchString(msg) {
			continue
		}

		if e = Es.Find(msg); e != nil {
			ce = e
			e.Metric.Inc()
			o = NewOops(s.TS, e.Name, e.Severity, s.Host)
			ol = append(ol, o)
		}
		if ce != nil && o != nil {
			matched, d := ce.Parse(msg)
			if matched {
				o.AddLogRaw(msg)
				o.Parse(d)
			}
		}
	}
	return ol
}
