package coroner

import (
	"fmt"
	"io/ioutil"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestOopsParse(t *testing.T) {
	var (
		tests = map[string]*Oops{
			"gpf": &Oops{
				Name:       "gpf",
				Message:    "general protection fault: 0000 [#1] SMP NOPTI",
				Kernel:     "4.19.162-40",
				Command:    "yt-solomon-brid",
				Pid:        "15218",
				CPU:        "10",
				Extra:      map[string][]string{"": []string{"Not tainted"}, "bios": []string{"R22 07/21/2015"}},
				StackTrace: []string{"__bfq_deactivate_entity+0xe3/0x1d0", "bfq_deactivate_entity+0x53/0xc0"},
				Rip:        "0010:bfq_insert+0x14/0x60",
				HWName:     "GIGABYTE T17I-138/GA-7PTSH",
			},
			"bug": &Oops{
				Name:       "bug",
				Message:    "kernel BUG at mm/huge_memory.c:2652!",
				Kernel:     "4.14.78-29",
				Command:    "backctld",
				Pid:        "794002",
				CPU:        "10",
				Extra:      map[string][]string{"": []string{"Not tainted"}, "bios": []string{"3.0c 10/30/2014"}},
				StackTrace: []string{"?find_get_entry+0x20/0x140", "shmem_unused_huge_shrink+0x184/0x3f0"},
				Rip:        "0010:split_huge_page_to_list+0x7b5/0x8d0",
				HWName:     "Supermicro X9DRW/X9DRW",
			},
			"workqueue_lockup": &Oops{
				Name:    "workqueue_lockup",
				Message: "BUG: workqueue lockup - pool cpus=29 node=1 flags=0x0 nice=0 stuck for 41s!",
				Extra:   map[string][]string{},
			},
			"bert_error": &Oops{
				Name:    "bert_error",
				Message: "BERT: Error records from previous boot:",
				Extra:   map[string][]string{},
			},
			"hardware_error": &Oops{
				Name:    "hardware_error",
				Message: "Error 0, type: fatal",
				CPU:     "0xe",
				Extra:   map[string][]string{"fru_text": []string{"ProcessorError"}, "section_type": []string{"IA32/X64 processor error"}, "error_structure_type": []string{"cache error"}, "MSR_Address": []string{"0xc0002051"}},
			},
		}
		testCount = map[string]int{
			"gpf":              1,
			"bug":              1,
			"workqueue_lockup": 1,
			"bert_error":       1,
			"hardware_error":   3,
		}
	)

	for n, exp := range tests {
		buf, err := ioutil.ReadFile("testdata/" + n + ".txt")
		require.NoError(t, err)
		o := OopsParse(&Session{Data: buf})
		fmt.Println(o)
		require.Equal(t, testCount[n], len(o))
		act := o[0]
		require.Equal(t, exp.Name, act.Name)
		require.Equal(t, exp.Message, act.Message)
		require.Equal(t, exp.Kernel, act.Kernel)
		require.Equal(t, exp.Command, act.Command)
		require.Equal(t, exp.Pid, act.Pid)
		require.Equal(t, exp.CPU, act.CPU)
		require.Equal(t, exp.Dev, act.Dev)
		require.Equal(t, exp.Extra, act.Extra)
		require.Equal(t, exp.StackTrace, act.StackTrace)
		require.Equal(t, exp.Rip, act.Rip)
		require.Equal(t, exp.WorkQueue, act.WorkQueue)
		require.Equal(t, exp.HWName, act.HWName)
	}
}
