package coroner

import (
	"context"
	"fmt"
	"io/ioutil"
	"net"
	"net/http"
	"strconv"
	"strings"
	"sync"
	"time"

	ctl "a.yandex-team.ru/infra/rsm/coroner/internal/app/coronerctl"
	"a.yandex-team.ru/library/go/core/log/zap"
)

const (
	maxBufferSize = 65536
)

type UDPSrv struct {
	port  int
	queue chan *Session
	l     *zap.Logger
}

func NewUDPSrv(port int, queue chan *Session, l *zap.Logger) (*UDPSrv, error) {
	//TODO: check udp port
	return &UDPSrv{
		port:  port,
		queue: queue,
		l:     l,
	}, nil
}

func (s *UDPSrv) Run(ctx context.Context, wg *sync.WaitGroup) {
	s.l.Infof("starting UDP Server")
	defer wg.Done()

	conn, err := net.ListenPacket("udp", fmt.Sprintf(":%d", s.port))
	if err != nil {
		panic(err)
	}
	defer conn.Close()

	go func() {
		for {
			select {
			case <-ctx.Done():
				return
			default:
				buf := make([]byte, maxBufferSize)
				n, addr, err := conn.ReadFrom(buf)
				if err != nil {
					s.l.Errorf("%s", err)
					continue
				}
				s.l.Debugf("got connection %s", addr)
				host, err := lookupAddr(addr)
				if err != nil {
					s.l.Errorf("%s", err)
					host = addr.String()
				}
				s.queue <- &Session{time.Now().UnixMicro(), addr.String(), host, buf[0:n]}
			}
		}
	}()

	<-ctx.Done()
	s.l.Infof("UDPSrv exit by ctx\n")
}

type HTTPSrv struct {
	s     *http.Server
	queue chan *Session
	l     *zap.Logger
}

func NewHTTPSrv(port int, queue chan *Session, l *zap.Logger) (*HTTPSrv, error) {
	//TODO: check http and tcp port
	mux := http.NewServeMux()
	s := &http.Server{
		Addr:    fmt.Sprintf(":%d", port),
		Handler: mux,
	}
	mux.HandleFunc("/ping", pingHandler(l))
	mux.HandleFunc("/api/v1/new_oops", newOopsHandler(queue, l))
	mux.HandleFunc("/api/v1/raw", rawHandler(l))
	return &HTTPSrv{
		s:     s,
		queue: queue,
		l:     l,
	}, nil
}

func (h *HTTPSrv) Run(ctx context.Context, wg *sync.WaitGroup) {
	h.l.Infof("starting HTTPSrv")
	defer wg.Done()

	go func() {
		if err := h.s.ListenAndServe(); err != nil {
			h.l.Fatalf("%s", err)
		}
	}()

	<-ctx.Done()
	if err := h.s.Shutdown(context.Background()); err != nil {
		h.l.Errorf("%s", err)
	}
	h.l.Infof("HttpSrv exit by ctx")
}

func pingHandler(l *zap.Logger) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		respDo(w, &Response{200, "pong"}, l)
	})
}

func newOopsHandler(q chan *Session, l *zap.Logger) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		var (
			ts   = time.Now().UnixMicro()
			host string
		)

		if tsS := getOpt("ts", r); tsS != "" {
			tsI, err := strconv.ParseInt(tsS, 10, 64)
			if err != nil {
				respDo(w, &Response{400, fmt.Sprintf("can't parse ts: %s", err)}, l)
				return
			}
			ts = tsI
		}

		if host = getOpt("host", r); host == "" {
			respDo(w, &Response{400, "get param host requred"}, l)
			return
		}

		data, err := ioutil.ReadAll(r.Body)
		if err != nil {
			l.Errorf("%s", err)
			return
		}
		if len(data) == 0 {
			respDo(w, &Response{400, "data is null in POST Request"}, l)
			return
		}
		respDo(w, &Response{
			Code: 200,
			Msg:  fmt.Sprintf("ts:%d, host:%s, data:%d bytes", ts, host, len(data)),
		}, l)
		q <- &Session{time.Now().UnixMicro(), "", host, data}
	})
}

func rawHandler(l *zap.Logger) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		var (
			host string
			tb   int64
		)

		if tbS := getOpt("tail_bytes", r); tbS != "" {
			tbI, err := strconv.ParseInt(tbS, 10, 64)
			if err != nil {
				respDo(w, &Response{400, fmt.Sprintf("can't parse tail_bytes: %s", err)}, l)
				return
			}
			tb = tbI
		} else {
			tb = 0
		}

		if host = getOpt("host", r); host == "" {
			respDo(w, &Response{400, "get param host requred"}, l)
			return
		}
		s, err := ctl.MakeRawQuery(tb, host)
		if err != nil {
			respDo(w, &Response{400, fmt.Sprintf("failed to make query: %s", err)}, l)
			return
		}
		respDo(w, &Response{200, s}, l)
	})
}

type Response struct {
	Code int
	Msg  string
}

func respDo(w http.ResponseWriter, r *Response, l *zap.Logger) {
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(r.Code)
	if _, err := fmt.Fprintln(w, r.Msg); err != nil {
		l.Errorf("%s", err)
	}
}

func lookupAddr(addr net.Addr) (host string, err error) {
	host, _, err = net.SplitHostPort(addr.String())
	if err != nil {
		return
	}
	names, err := net.LookupAddr(host)
	if err != nil {
		return
	}
	host = strings.TrimRight(names[0], ".")
	return
}

func getOpt(k string, r *http.Request) (v string) {
	d, ok := r.URL.Query()[k]
	if !ok || len(d) == 0 {
		return
	}
	v = d[0]
	return
}
