package coroner

import (
	"context"
	"encoding/json"
	"os"
	"path/filepath"
	"sync"

	"a.yandex-team.ru/infra/rsm/coroner/pkg/lb"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/log/corelogadapter"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log/zap"
)

const (
	LBEndPoint = "logbroker.yandex.net"
	LBDB       = "/Root"
)

type LBStorage struct {
	l    *zap.Logger
	coLB *lb.Producer
	crLB *lb.Producer
}

func NewLBStorage(l *zap.Logger, oq chan json.Marshaler, rq chan json.Marshaler, prfx string) *LBStorage {
	fqdn, err := os.Hostname()
	if err != nil {
		l.Fatalf("%s", err)
	}

	token, ok := os.LookupEnv("LB_TOKEN")
	if ok {
		l.Debugf("Use LB_TOKEN")
	}
	cred := ydb.AuthTokenCredentials{AuthToken: token}

	oopsTopic := filepath.Join(prfx, "oops")
	rawTopic := filepath.Join(prfx, "raw")

	return &LBStorage{
		l: l,
		coLB: lb.NewProducer(l, oq, &persqueue.WriterOptions{
			Endpoint:       LBEndPoint,
			Credentials:    cred,
			Database:       LBDB,
			Logger:         corelogadapter.New(l),
			Topic:          oopsTopic,
			SourceID:       []byte(fqdn),
			Codec:          persqueue.Zstd,
			RetryOnFailure: true,
		}),
		crLB: lb.NewProducer(l, rq, &persqueue.WriterOptions{
			Endpoint:       LBEndPoint,
			Credentials:    cred,
			Database:       LBDB,
			Logger:         corelogadapter.New(l),
			Topic:          rawTopic,
			SourceID:       []byte(fqdn),
			Codec:          persqueue.Zstd,
			RetryOnFailure: true,
		}),
	}
}

func (s *LBStorage) Run(ctx context.Context, wg *sync.WaitGroup) {
	defer wg.Done()

	wg.Add(1)
	go func() {
		defer wg.Done()
		s.l.Errorf("LBoops exiting by %s", s.coLB.Run(ctx))
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		s.l.Errorf("LBraw exiting by %s", s.crLB.Run(ctx))
	}()

	<-ctx.Done()
	wg.Wait()
}
