package cmd

import (
	"bytes"
	"fmt"
	"io"
	"os"
	"strings"

	"a.yandex-team.ru/infra/rsm/coroner/internal/app/coroner"
	ctl "a.yandex-team.ru/infra/rsm/coroner/internal/app/coronerctl"
	log "a.yandex-team.ru/infra/rsm/coroner/internal/logger"

	"github.com/spf13/cobra"
)

var (
	cmd = &cobra.Command{
		Use: "parse",
		Run: func(cmd *cobra.Command, args []string) { runParseCmd() },
	}
)

func init() {
	cmd.Flags().StringVarP(&opts.Columns, "columns", "c", "dt,h,p,n,m,k,rip,s",
		"comma-separated list of columns, (id, [d]ate, [t]ime, dt, [h]ost, [p]rj, s[e]verity, [n]ame, [m]essage, [k]ernel, [s]tacktrace, [ex]tra)")
	rootCmd.AddCommand(cmd)
}

func runParseCmd() {
	columnsSelected := ctl.ColumnParser(opts.Columns, nil, false)
	if len(columnsSelected) == 0 {
		log.L.Fatalf("columns not selected, please specify -c")
	}
	buf := bytes.NewBuffer(make([]byte, 0))
	_, err := io.CopyBuffer(buf, os.Stdin, make([]byte, 100*1024*1024))

	if err != nil {
		log.L.Fatalf("%s", err)
	}
	table := ctl.NewTable(os.Stdout)
	table.SetHeader(columnsSelected)
	tableData := func() [][]string {
		result := [][]string{}
		s := &coroner.Session{
			TS:   0,
			Addr: "",
			Data: buf.Bytes(),
		}
		for _, o := range coroner.OopsParse(s) {
			row := []string{}
			for _, c := range columnsSelected {
				switch c {
				case "svrt":
					row = append(row, string(o.Severity))
				case "name":
					row = append(row, o.Name)
				case "message":
					row = append(row, o.Message)
				case "kernel":
					row = append(row, o.Kernel)
				case "command":
					row = append(row, o.Command)
				case "pid":
					row = append(row, o.Pid)
				case "cpu":
					row = append(row, o.CPU)
				case "dev":
					row = append(row, o.Dev)
				case "stacktrace":
					row = append(row, strings.ReplaceAll(o.StackTraceStr, ";", " "))
				case "rip":
					row = append(row, o.Rip)
				case "workqueue":
					row = append(row, o.WorkQueue)
				case "hwname":
					row = append(row, o.HWName)
				case "extra":
					row = append(row, mapToStr(o.Extra))
				case "lograw":
					row = append(row, strings.Join(o.LogRaw, "\n"))
				default:
					row = append(row, "none")
				}
			}
			result = append(result, row)
		}
		return result
	}
	table.AppendBulk(tableData())
	table.Render()
}

func mapToStr(m map[string][]string) string {
	result := new(bytes.Buffer)
	for k, v := range m {
		fmt.Fprintf(result, "%s=\"%s\" \n", k, v)
	}
	return result.String()
}

//func readFromFile(fp string, w io.Writer) error {
//	f, err := os.Open(fp)
//	if err != nil {
//		return err
//	}
//	defer f.Close()
//
//  _, err = io.Copy(w, f)
//	return err
//}

//func readFromKmsg(fp string, w io.Writer) error {
//	// https://www.kernel.org/doc/Documentation/ABI/testing/dev-kmsg
//	fd, err := syscall.Open(fp, syscall.O_RDONLY|syscall.O_NONBLOCK, 0)
//	if err != nil {
//			return err
//	}
//	defer syscall.Close(fd)
//
//	buf := make([]byte, 8192)
//	for {
//			count, err := syscall.Read(fd, buf)
//			if err != nil {
//				if err == syscall.EAGAIN {
//					break
//				}
//				return err
//			}
//			w.Write(buf[:count])
//	}
//	return nil
//}
