package cmd

import (
	"fmt"
	"os"
	"strings"

	"a.yandex-team.ru/infra/rsm/coroner/internal/app/coroner"
	ctl "a.yandex-team.ru/infra/rsm/coroner/internal/app/coronerctl"
	log "a.yandex-team.ru/infra/rsm/coroner/internal/logger"
	"a.yandex-team.ru/library/go/core/buildinfo"

	"github.com/spf13/cobra"
)

var (
	opts = &ctl.Opts{}

	rootCmd = &cobra.Command{
		Use: "coronerctl",
		Long: `
Wiki: https://wiki.yandex-team.ru/kernel/#coroner
Docs: https://a.yandex-team.ru/arc/trunk/arcadia/infra/rsm/coroner/README.md

Columns:
  id - id oops
  d, date - date in format ""
  t, time - time in format ""
  dt - date + time
  h, host - fqdn
  p, prj - wall-e prj
  e, svrt - severity
  n, name - name OOPS
  m, message - full parsed message
  k, kernel - kernel version
  s, stack, stacktrace - OOPS's stacktrace
  c, count - special field when use --sumby arg

Examples:
  1) coronerctl -l 10 -t 3d --sumby n -c d,e,n --sortby D,c -E ^INFO
                    |     |         |     |              |       |
                    |     |         |     |              |       |_ exclude "INFO" severity
                    |     |         |     |              |_ sort by field "date" DESC and field "count" ASC
                    |     |         |     |_ show columns "date", "severity", "name"
                    |     |         |_ summary by column "name"
                    |     |_ last 3 days
                    |_ limit 10 lines


  2) coronerctl -l 10 -t 15.10-16.10 --sumby name -c e,n,m -N ^mce_hardware,^task_lockup -M DRIVER_SENSE
                             |                                         |                         |
                             |                                         |                         |_ OOPS message contains DRIVER_SENSE
                             |                                         |_ exclude OOPS names "mce_hardware","task_lockup"
                             |_ from 15 Oct 2020 to 16 Oct 2020

  3) coronerctl -l 10 -c d,h,n,s -S mlx --sumby s --sortby C
                                    |                      |_ sort by "count" DESC
                                    |
                                    |_ OOPS stacktrace contains "mlx"
`,
		Version: buildinfo.Info.ProgramVersion,
		PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
			return log.SetupLogger("", opts.Level)
		},
		Run: func(cmd *cobra.Command, args []string) { runRootCmd() },
	}
)

func Execute() {
	rootCmd.Flags().StringVar(&opts.Use, "db", ctl.CHYTName, "set YT DB path")
	rootCmd.Flags().StringVar(&opts.From, "table", ctl.OopsTable, "set YT Table path")
	rootCmd.Flags().IntVarP(&opts.Limit, "limit", "l", 1000, "maximum number of returned lines")
	rootCmd.Flags().StringVarP(&opts.T, "time", "t", "3d", "filter by time (3d-now, 1m, 22.04-18.09T15:05)")
	rootCmd.Flags().StringVarP(&opts.Output, "output", "o", "table", "output format (table, json, yaml)")
	rootCmd.PersistentFlags().CountVarP(&opts.Level, "debug", "d", "-ddd - debug, -dd - info, -d - error. Default -d")
	rootCmd.Flags().BoolVar(&opts.Dryrun, "dryrun", false, "dryrun mode. Only makes request and print then")
	rootCmd.Flags().StringVarP(&opts.ID, "id", "I", "", "filter by id")
	rootCmd.Flags().StringVarP(&opts.Host, "host", "H", "", "filter by comma-separated list of host (sas1-3333.search.yandex.net,)")
	rootCmd.Flags().StringVarP(&opts.Prj, "prj", "P", "", "filter by comma-separated list of wall-e projects (yp-iss-iva,)")
	rootCmd.Flags().StringVarP(&opts.Svrt, "svrt", "E", "", "filter by comma-separated list of severity (CRIT, ERROR, WARN, INFO)")
	rootCmd.Flags().StringVarP(&opts.Name, "name", "N", "", fmt.Sprintf("filter by comma-separated list of names (%s)", strings.Join(coroner.Es.Names(), ", ")))
	rootCmd.Flags().StringVarP(&opts.Msg, "msg", "M", "", "filter by message")
	rootCmd.Flags().StringVarP(&opts.Kernel, "kernel", "K", "", "filter by kernel")
	rootCmd.Flags().StringVarP(&opts.Stack, "stack", "S", "", "filter by stacktrace")
	rootCmd.Flags().StringVarP(&opts.Rip, "rip", "R", "", "filter by RIP")
	rootCmd.Flags().StringVarP(&opts.Wq, "wq", "W", "", "filter by WorkQueue")
	rootCmd.Flags().StringVar(&opts.Hw, "hw", "", "filter by Hardware")
	rootCmd.Flags().StringVarP(&opts.Columns, "columns", "c", "dt,h,p,n,m,k,rip,s",
		"comma-separated list of columns, (id, [d]ate, [t]ime, dt, [h]ost, [p]rj, s[e]verity, [n]ame, [m]essage, [k]ernel, [s]tacktrace, [ex]trai, cpu)")
	rootCmd.Flags().StringVar(&opts.Sumby, "sumby", "", "summary (func COUNT()) by field")
	rootCmd.Flags().StringVar(&opts.Sortby, "sortby", "", "sort by column name. For DESC func use Uppercase, like Host or H")

	if buildinfo.Info.ProgramVersion == "" {
		rootCmd.Version = "N/A"
	} else {
		rootCmd.Version = buildinfo.Info.ProgramVersion
	}
	if err := rootCmd.Execute(); err != nil {
		fmt.Println(err)
	}
}

func runRootCmd() {
	if err := ctl.Run(opts, os.Stdout); err != nil {
		log.L.Errorf("%s", err)
		os.Exit(1)
	}
}
