package coronerctl

import (
	"fmt"
	"io"
	"strings"

	log "a.yandex-team.ru/infra/rsm/coroner/internal/logger"
)

const (
	EXTRATABLE = "//home/runtimecloud/coroner/walle_fqdn_cpu_mapping"
	TMPLSTR    = `
                USE {{ .Use }};
                SELECT {{ join .Select ", " }}
                FROM
                        (SELECT {{ join .SubSelect ", " }}
                        FROM concatYtTablesRange( '{{ .From }}/1d', '{{ .FromTable }}', '{{ .ToTable }}' )
                        UNION DISTINCT
                        SELECT {{ join .SubSelect ", " }}
                        FROM concatYtTablesRange( '{{ .From }}/stream/5min', '{{ .FromTable }}', '{{ .ToTableMin }}' )) as t1
                LEFT OUTER JOIN
                (SELECT fqdn,dc,walle_project,cpu_model FROM "` + EXTRATABLE + `") as t2
                        on t1.host == t2.fqdn
                {{if .Where}}WHERE {{ join .Where " AND " }}{{end}}
                {{if .Group}}GROUP BY {{ join .Group ", " }}{{end}}
                {{if .Order}}ORDER BY {{ join .Order ", " }}{{end}}
                LIMIT {{ .Limit }}
        `
	CHYTName  = "chyt.hahn/ch_coroner"
	OopsTable = "//home/logfeller/logs/rtc-coroner-oops-log"
	RawTable  = "//home/logfeller/logs/rtc-coroner-raw-log"
)

var (
	OutterColumns = map[string]string{
		"id":         "id",
		"d":          "date",
		"date":       "date",
		"ts":         "ts",
		"t":          "time",
		"time":       "time",
		"dt":         "datetime",
		"datetime":   "datetime",
		"h":          "host",
		"host":       "host",
		"p":          "prj",
		"prj":        "prj",
		"e":          "svrt",
		"svrt":       "svrt",
		"severity":   "svrt",
		"n":          "name",
		"name":       "name",
		"m":          "message",
		"message":    "message",
		"k":          "kernel",
		"kernel":     "kernel",
		"cmd":        "command",
		"pid":        "pid",
		"cpu":        "cpu",
		"dev":        "dev",
		"ex":         "extra",
		"extra":      "extra",
		"s":          "stacktrace",
		"stack":      "stacktrace",
		"stacktrace": "stacktrace",
		"rip":        "rip",
		"wq":         "workqueue",
		"workqueue":  "workqueue",
		"hw":         "hwname",
		"hwname":     "hwname",
		"lr":         "lograw",
		"lograw":     "lograw",
		"c":          "count",
		"count":      "count",
		"data":       "data",
	}

	Columns = map[string]string{
		"date":       "formatDateTime(toDateTime64(ts/1000000, 6),'%d.%m')",
		"time":       "formatDateTime(toDateTime64(ts/1000000, 6),'%T')",
		"datetime":   "formatDateTime(toDateTime64(ts/1000000, 6),'%d.%m.%yT%T')",
		"stacktrace": "stacktracestr",
		"prj":        "walle_project",
		"svrt":       "severity",
	}

	InnerColumns = map[string]string{
		"id":         "id",
		"d":          "ts",
		"date":       "ts",
		"t":          "ts",
		"time":       "ts",
		"dt":         "ts",
		"datetime":   "ts",
		"h":          "host",
		"host":       "host",
		"e":          "severity",
		"svrt":       "severity",
		"severity":   "severity",
		"n":          "name",
		"name":       "name",
		"m":          "message",
		"message":    "message",
		"k":          "kernel",
		"kernel":     "kernel",
		"cmd":        "command",
		"pid":        "pid",
		"cpu":        "cpu",
		"dev":        "dev",
		"ex":         "extra",
		"extra":      "extra",
		"s":          "stacktracestr",
		"stack":      "stacktracestr",
		"stacktrace": "stacktracestr",
		"rip":        "rip",
		"wq":         "workqueue",
		"workqueue":  "workqueue",
		"hw":         "hwname",
		"hwname":     "hwname",
		"lr":         "lograw",
		"lograw":     "lograw",
		"c":          "count",
		"count":      "count",
		"data":       "data",
	}
)

func ColumnParser(s string, qd *QueryData, withQD bool) []string {
	var (
		columnsSelected []string
	)
	for _, i := range strings.Split(s, ",") {
		if c, ok := OutterColumns[i]; ok {
			columnsSelected = append(columnsSelected, c)
			if withQD {
				if a, ok := Columns[c]; ok {
					qd.UpdateSelect(fmt.Sprintf("%s as %s", a, c))
				} else {
					qd.UpdateSelect(c)
				}
			}
		} else {
			log.L.Warnf("column %s not found. Skip it\n", i)
		}
		if c, ok := InnerColumns[i]; withQD && ok {
			qd.UpdateSubSelect(c)
		}
	}
	return columnsSelected
}

func Run(opts *Opts, w io.Writer) error {
	qd, err := NewQD(TMPLSTR).ParseOpts(opts)
	if err != nil {
		return err
	}

	if err := qd.FetchData(); err != nil {
		return err
	}
	log.L.Debugf("Doing report\n")
	if err := qd.Report(w); err != nil {
		return err
	}
	return nil
}

func makeIN(qd *QueryData, s, col string) []string {
	result := []string{}
	if s == "" {
		return result
	}
	if _, ok := InnerColumns[col]; ok {
		qd.UpdateSubSelect(col)
	}
	in, notin := arrSplitBy(strings.Split(s, ","), "^")
	if len(in) > 0 {
		result = append(result, fmt.Sprintf("%s IN ('%s')", col, strings.Join(in, "','")))
	}
	if len(notin) > 0 {
		result = append(result, fmt.Sprintf("%s NOT IN ('%s')", col, strings.Join(notin, "','")))
	}
	return result
}

func makeLike(qd *QueryData, s, col string) []string {
	result := []string{}
	if s == "" {
		return result
	}
	if _, ok := InnerColumns[col]; ok {
		qd.UpdateSubSelect(col)
	}
	in, notin := arrSplitBy(strings.Split(s, ","), "^")
	if len(in) > 0 {
		result = append(result, fmt.Sprintf("%s LIKE '%s'", col, "%"+in[0]+"%"))
	}
	if len(notin) > 0 {
		result = append(result, fmt.Sprintf("%s NOT LIKE '%s'", col, "%"+notin[0]+"%"))
	}
	return result
}

func arrSplitBy(a []string, s string) ([]string, []string) {
	y := []string{}
	n := []string{}
	for _, i := range a {
		if strings.HasPrefix(i, s) {
			y = append(y, i[len(s):])
		} else {
			n = append(n, i)
		}
	}
	return n, y
}

func isExistInArr(s string, a []string) bool {
	for _, i := range a {
		if s == i {
			return true
		}
	}
	return false
}
