package coronerctl

import (
	"bytes"
	"errors"
	"fmt"
	"io"
	"strings"
	"text/template"
	"time"

	log "a.yandex-team.ru/infra/rsm/coroner/internal/logger"
	"a.yandex-team.ru/infra/rsm/coroner/pkg/yql"

	"github.com/valyala/fastjson"
)

type QueryData struct {
	Use        string
	SubSelect  []string
	Select     []string
	From       string
	Where      []string
	Group      []string
	Order      []string
	Limit      int
	Tmpl       *template.Template
	Query      bytes.Buffer
	Data       []*fastjson.Value
	FromTable  string
	ToTable    string
	ToTableMin string
	Output     string
}

func NewQD(tmplstr string) *QueryData {
	return &QueryData{
		Tmpl: template.Must(template.New("query").Funcs(template.FuncMap{
			"join": strings.Join,
		}).Parse(tmplstr)),
	}
}

func (d *QueryData) UpdateWhere(s []string) {
	if len(s) > 0 {
		d.Where = append(d.Where, s...)
	}
}

func (d *QueryData) UpdateWhereTime(s []string) {
	if len(d.Where) < 2 {
		d.Where = append(d.Where, s...)
	} else {
		d.Where[0] = s[0]
		d.Where[1] = s[1]
	}
}

func (d *QueryData) UpdateSelect(s string) {
	if ok := isExistInArr(s, d.Select); !ok {
		d.Select = append(d.Select, s)
	}
}

func (d *QueryData) UpdateSubSelect(s string) {
	if ok := isExistInArr(s, d.SubSelect); !ok {
		d.SubSelect = append(d.SubSelect, s)
	}
}

func (d *QueryData) FetchData() error {
	yqlc, err := yql.NewClient(log.L)
	if err != nil {
		return err
	}
	q := yqlc.Query(yql.EngineCH, d.From, d.Query.String())
	err = q.Run("RUN")
	if err != nil {
		return err
	}
	d.Data = q.Data
	return nil
}

func (d *QueryData) GetDataWithColumnNames() *fastjson.Value {
	var fa fastjson.Arena
	result := fa.NewArray()

	for x, r := range d.Data[0].Get("Data").GetArray() {
		fo := fa.NewObject()
		for i, v := range r.GetArray() {
			if v.Type() == fastjson.TypeArray {
				v = v.GetArray()[0]
			}
			fo.Set(string(d.Data[0].Get("Type").GetStringBytes("1", "1", fmt.Sprintf("%d", i), "0")), v)
		}
		result.SetArrayItem(x, fo)
	}
	return result
}

func (d *QueryData) Report(w io.Writer) error {
	log.L.Debugf("Make Report")
	switch d.Output {
	case "json":
		return DoJSONReport(d.GetDataWithColumnNames(), w)
	case "yaml":
		return DoYAMLReport(d.GetDataWithColumnNames(), w)
	case "table":
		return DoTableReport(d.Data, w)
	case "raw":
		return DoRawReport(d, w)
	case "text":
		return DoTextReport(d.GetDataWithColumnNames(), w)
	case "ui":
		return DoReverseReport(d.Data, w)
	default:
		return fmt.Errorf("%s format not supported", d.Output)
	}
}

func (d *QueryData) SetTime(s string) error {
	d.UpdateSubSelect("ts")
	timeWhere := make([]string, 0, 2)
	tr, err := StrToTimeRange(s)
	if err != nil {
		return err
	}
	if len(tr) < 2 {
		tmp := time.Now().AddDate(0, 0, 1)
		tr = append(tr, &tmp)
	}
	timeWhere = append(timeWhere, fmt.Sprintf("ts >= '%d'", tr[0].UnixMicro()), fmt.Sprintf("ts <= '%d'", tr[1].UnixMicro()))
	d.UpdateWhereTime(timeWhere)
	if DateComp(*(tr[0]), time.Now().AddDate(0, 0, -1)) || DateComp(*(tr[0]), time.Now()) {
		tmp := time.Now().AddDate(0, 0, -2)
		tr[0] = &tmp
	}
	if DateComp(*(tr[1]), time.Now()) || DateComp(*(tr[1]), time.Now().AddDate(0, 0, -1)) {
		tmp := time.Now().AddDate(0, 0, 1)
		tr[1] = &tmp
	}
	sr, err := TimeRangeToStr(tr)
	if err != nil {
		return err
	}
	d.FromTable = sr[0]
	d.ToTable = sr[1]
	d.ToTableMin = time.Now().AddDate(0, 0, 1).Format("2006-01-02") //TODO make some logic for events -t d1-d2 d2 not in 5min tables
	return nil
}

func (d *QueryData) DoQuery() error {
	d.Query.Reset()
	if err := d.Tmpl.Execute(&d.Query, d); err != nil {
		return err
	}

	log.L.Debugf("made request: %s\n", d.Query.String())

	if err := d.FetchData(); err != nil {
		return err
	}
	log.L.Debugf("Data feched\n")
	return nil
}

func (d *QueryData) ParseOpts(opts *Opts) (*QueryData, error) {
	d.Output = opts.Output
	d.Use = opts.Use
	d.From = opts.From
	d.Limit = opts.Limit
	d.UpdateSubSelect("id")
	d.UpdateSubSelect("host")
	err := d.SetTime(opts.T)
	if err != nil {
		return nil, err
	}
	columnsSelected := ColumnParser(opts.Columns, d, true)
	if len(columnsSelected) == 0 {
		return nil, errors.New("columns not selected, please specify -c")
	}

	d.UpdateWhere(makeIN(d, opts.ID, "id"))
	d.UpdateWhere(makeIN(d, opts.Host, "host"))
	d.UpdateWhere(makeIN(d, opts.Prj, "walle_project"))
	d.UpdateWhere(makeIN(d, opts.Svrt, "severity"))
	d.UpdateWhere(makeIN(d, opts.Name, "name"))
	d.UpdateWhere(makeLike(d, opts.Msg, "message"))
	d.UpdateWhere(makeLike(d, opts.Kernel, "kernel"))
	d.UpdateWhere(makeLike(d, opts.Stack, "stacktrace"))
	d.UpdateWhere(makeLike(d, opts.Rip, "rip"))
	d.UpdateWhere(makeLike(d, opts.Wq, "workqueue"))
	d.UpdateWhere(makeLike(d, opts.Hw, "hwname"))

	if opts.Sumby != "" {
		d.Group = columnsSelected
		if n, ok := OutterColumns[opts.Sumby]; ok {
			if a, ok := Columns[n]; ok {
				d.UpdateSelect(fmt.Sprintf("count(%s) as count", a))
			} else {
				d.UpdateSelect(fmt.Sprintf("count(%s) as count", n))
			}
			columnsSelected = append(columnsSelected, "count")
			d.Order = append(d.Order, "count DESC")
		} else {
			log.L.Warnf("column %s not found. Skip it\n", opts.Sumby)
		}
	}

	if opts.Sortby != "" {
		d.Order = []string{}
		for _, i := range strings.Split(opts.Sortby, ",") {
			if n, ok := OutterColumns[strings.ToLower(i)]; ok {
				if isExistInArr(n, columnsSelected) {
					if n == "svrt" {
						n = "CASE svrt WHEN 'CRIT' THEN 1 WHEN 'ERROR' THEN 2 WHEN 'WARN' THEN 3 ELSE 4 END"
					}
					if strings.ToUpper(string(i[0])) == string(i[0]) {
						n = fmt.Sprintf("%s DESC", n)
					}
					d.Order = append(d.Order, n)
				} else {
					log.L.Warnf("column %s not selected, please specify -c. Skip it\n", i)
				}
			} else {
				log.L.Warnf("column %s not found. Skip it\n", i)
			}
		}
	}

	if err := d.Tmpl.Execute(&d.Query, d); err != nil {
		return nil, err
	}

	log.L.Debugf("made request: %s\n", d.Query.String())
	if opts.Dryrun {
		return d, nil
	}
	return d, nil
}
