package coronerctl

import (
	"fmt"
	"io"
	"strconv"
	"strings"
	"time"

	log "a.yandex-team.ru/infra/rsm/coroner/internal/logger"
	"github.com/olekukonko/tablewriter"
	"github.com/valyala/fastjson"
	"gopkg.in/yaml.v2"
)

var (
	TextData string
)

func NewTable(w io.Writer) *tablewriter.Table {
	table := tablewriter.NewWriter(w)
	//table.SetBorders(tablewriter.Border{Top: false})
	table.SetAutoWrapText(true)
	table.SetAlignment(tablewriter.ALIGN_LEFT)
	table.SetRowLine(true)
	return table
}

func DoYAMLReport(fv *fastjson.Value, w io.Writer) error {
	var jObj interface{}
	err := yaml.Unmarshal(fv.MarshalTo(nil), &jObj)
	if err != nil {
		return err
	}
	yObj, err := yaml.Marshal(jObj)
	if err != nil {
		return err
	}
	if _, err := fmt.Fprintf(w, "%s", yObj); err != nil {
		return err
	}
	return nil
}

func DoJSONReport(fv *fastjson.Value, w io.Writer) error {
	if _, err := fmt.Fprintf(w, "%s\n", fv.MarshalTo(nil)); err != nil {
		return err
	}
	return nil
}

func DoTableReport(data []*fastjson.Value, w io.Writer) error {
	table := NewTable(w)

	header := []string{}
	for _, s := range data[0].Get("Type").GetArray("1", "1") {
		header = append(header, string(s.GetStringBytes("0")))
	}
	table.SetHeader(header)

	tableData := func() [][]string {
		result := [][]string{}
		for _, r := range data[0].GetArray("Data") {
			row := []string{}
			for _, v := range r.GetArray() {
				if v.Type() == fastjson.TypeArray {
					v = v.GetArray()[0]
				}
				// tmp hack for stacktraceStr column
				col := strings.ReplaceAll(string(v.GetStringBytes()), ";", " \n")
				row = append(row, col)
			}
			result = append(result, row)
		}
		return result
	}

	table.AppendBulk(tableData())
	table.Render()
	return nil
}

func parseJSON(data []*fastjson.Value) (bool, string, [][]string, error) {
	truncated := data[0].GetBool("Truncated")
	pd := [][]string{}
	infoStr := ""
	var (
		ts  int64
		err error
	)
	for _, r := range data[0].GetArray("Data") {
		pd = append(pd, []string{})
		var buf []string
		for _, v := range r.GetArray() {
			if v.Type() == fastjson.TypeArray {
				v = v.GetArray()[0]
			}
			col := strings.ReplaceAll(string(v.GetStringBytes()), ";", " \n")
			buf = append(buf, col)
		}
		slice := strings.Split(buf[len(buf)-1], "\n")
		infoStr = strings.Join(buf[:len(buf)-1], " ")
		ts, err = strconv.ParseInt(infoStr, 10, 64)
		if err != nil {
			return truncated, "", nil, err
		}
		for _, i := range slice[:len(slice)-1] {
			timeInfo := time.UnixMicro(ts)
			pd[len(pd)-1] = append(pd[len(pd)-1], fmt.Sprintf("%s %s\n", strings.Split(timeInfo.String(), ".")[0], i))
		}
	}
	if infoStr == "" {
		infoStr = "0"
	}
	ts, err = strconv.ParseInt(infoStr, 10, 64)
	if err != nil {
		return truncated, "", nil, err
	}
	return truncated, strconv.FormatInt(int64(ts), 10), pd, nil
}

func QueryWriter(data *QueryData, c chan error, w *io.PipeWriter) {
	truncated := true

	for {
		//exit after last part of query
		if !truncated {
			break
		}

		//parsing
		trc, tsString, pd, err := parseJSON(data.Data)
		truncated = trc
		if err != nil {
			c <- err
			break
		}

		//agregate for query processing time acceleration
		agregate := []string{}
		for i := 0; i < len(pd); i++ {
			for j := 0; j < len(pd[i]); j++ {
				agregate = append(agregate, pd[i][j])
			}
		}
		if _, err := fmt.Fprint(w, strings.Join(agregate, "")); err != nil {
			c <- err
			break
		}

		if truncated {
			ts, err := strconv.ParseInt(tsString, 10, 64)
			if err != nil {
				c <- err
				break
			}
			ts++
			//updating ts in query
			tmp := data.ToTable
			if err := data.SetTime(strconv.FormatInt(int64(ts), 10)); err != nil {
				c <- err
				break
			}
			data.ToTable = tmp
			//do next query
			if err := data.DoQuery(); err != nil {
				c <- err
				break
			}
		}
	}
	if err := w.Close(); err != nil {
		c <- err
	}
	c <- nil
}

func DoRawReport(data *QueryData, w io.Writer) error {
	r, pipeWriter := io.Pipe()
	timeout := time.After(1 * time.Second)
	c := make(chan error)
	go QueryWriter(data, c, pipeWriter)

	for {
		select {
		case err := <-c:
			if err != nil {
				return err
			}
			if err := r.Close(); err != nil {
				return err
			}
			return nil
		case <-timeout:
			_, err := io.Copy(w, r)
			if err != nil {
				return err
			}
		}
	}
}

func DoReverseReport(data []*fastjson.Value, w io.Writer) error {
	_, _, pd, err := parseJSON(data)
	if err != nil {
		return err
	}
	agregator := []string{}
	for i := 0; i < len(pd); i++ {
		for j, k := 0, len(pd[i])-1; j < k; j, k = j+1, k-1 {
			pd[i][j], pd[i][k] = pd[i][k], pd[i][j]
		}
		agregator = append(agregator, strings.Join(pd[i], ""))
	}
	if _, err := fmt.Fprintf(w, "%s\n", strings.Join(agregator, "")); err != nil {
		log.L.Debugf("Error")
		return err
	}
	return nil
}

func DoTextReport(data *fastjson.Value, w io.Writer) error {
	TextData = fmt.Sprintf("%s\n", data.MarshalTo(nil))
	_, err := fmt.Fprintf(w, "%s", TextData)
	return err
}
