package coronerctl

import (
	"fmt"
	"strconv"
	"strings"
	"time"
)

func TimeRangeToStr(tr []*time.Time) ([]string, error) {
	var result []string
	for _, s := range tr {
		result = append(result, s.Format("2006-01-02"))
	}
	return result, nil
}

func DateComp(a, b time.Time) bool {
	ay, am, ad := a.Date()
	by, bm, bd := b.Date()
	return ay == by && am == bm && ad == bd
}

func StrToTimeRange(s string) ([]*time.Time, error) {
	var result []*time.Time
	now := time.Now()
	for _, i := range strings.Split(s, "-") {
		if i == "now" {
			result = append(result, &now)
		} else if strings.ContainsAny(i, "S,M,h,d,m,y") {
			d, err := StrToTimeShift(now, i)
			if err != nil {
				return result, err
			}
			result = append(result, d)
		} else {
			d, err := StrToTime(now, i)
			if err != nil {
				return result, err
			}
			result = append(result, d)
		}
	}
	return result, nil
}

func StrToTimeShift(now time.Time, s string) (*time.Time, error) {
	var result time.Time
	num, err := strconv.ParseInt(s[:len(s)-1], 10, 32)
	if err != nil {
		return nil, err
	}
	switch string(s[len(s)-1]) {
	case "S":
		result = now.Add(time.Second * time.Duration(-num))
	case "M":
		result = now.Add(time.Minute * time.Duration(-num))
	case "h":
		result = now.Add(time.Hour * time.Duration(-num))
	case "d":
		result = now.AddDate(0, 0, int(-num))
	case "m":
		result = now.AddDate(0, int(-num), 0)
	case "y":
		result = now.AddDate(int(-num), 0, 0)
	default:
		return nil, fmt.Errorf("error parse: %s", s)
	}
	return &result, err
}

func StrToTime(now time.Time, s string) (*time.Time, error) {
	var result time.Time
	if t, err := time.Parse("15:04", s); err == nil {
		result = time.Date(now.Year(), now.Month(), now.Day(), t.Hour(), t.Minute(), 0, 0, time.Local)
	} else if t, err := time.Parse("02", s); err == nil {
		result = time.Date(now.Year(), now.Month(), t.Day(), now.Hour(), now.Minute(), 0, 0, time.Local)
	} else if t, err := time.Parse("02.01", s); err == nil {
		result = time.Date(now.Year(), t.Month(), t.Day(), now.Hour(), now.Minute(), 0, 0, time.Local)
	} else if t, err := time.Parse("02.01.2006", s); err == nil {
		result = time.Date(t.Year(), t.Month(), t.Day(), now.Hour(), now.Minute(), 0, 0, time.Local)
	} else if t, err := time.Parse("02.01.06", s); err == nil {
		result = time.Date(t.Year(), t.Month(), t.Day(), now.Hour(), now.Minute(), 0, 0, time.Local)
	} else if t, err := time.Parse("02.01T15", s); err == nil {
		result = time.Date(now.Year(), t.Month(), t.Day(), t.Hour(), now.Minute(), 0, 0, time.Local)
	} else if t, err := time.Parse("02.01T15:04", s); err == nil {
		result = time.Date(now.Year(), t.Month(), t.Day(), t.Hour(), t.Minute(), 0, 0, time.Local)
	} else if t, err := time.Parse("02.01.2006T15", s); err == nil {
		result = time.Date(t.Year(), t.Month(), t.Day(), t.Hour(), now.Minute(), 0, 0, time.Local)
	} else if t, err := time.Parse("02.01.2006T15:04", s); err == nil {
		result = time.Date(t.Year(), t.Month(), t.Day(), t.Hour(), t.Minute(), 0, 0, time.Local)
	} else if t, err := strconv.ParseInt(s, 10, 64); err == nil {
		result = time.UnixMicro(t)
	} else {
		return nil, fmt.Errorf("error parse: %s", s)
	}
	return &result, nil
}
