package yql

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"os"
	"os/user"
	"path/filepath"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/oauth"
	"github.com/valyala/fastjson"
)

// https://yql.yandex-team.ru/docs/yt/interfaces/http/
// https://yql.yandex-team.ru/docs/http/reference/

const (
	baseURL = "https://yql.yandex.net/api/v2"
	retries = 3

	EngineCH  Engine = "CLICKHOUSE"
	EngineSQL Engine = "SQL"

	// From https://oauth.yandex-team.ru/client/6b0f3ab9e5d74e998978bf9477764fba
	clientID     = "6b0f3ab9e5d74e998978bf9477764fba"
	clientSecret = "75f0ac55fb384f379ba7e9893bbda395"

	StatusCompleted Status = "COMPLETED"
	StatusPending   Status = "PENDING"
	StatusAborted   Status = "ABORTED"
	StatusError     Status = "ERROR"
	StatusUnknown   Status = "UNKNOWN"
)

type HTTPErr struct {
	Code int
	Text string
}

func (e HTTPErr) Error() string {
	return fmt.Sprintf("unexpected HTTP status code %d, response is %+v", e.Code, e.Text)
}

type Engine string
type Status string

type Client struct {
	baseURL   string
	token     string
	userAgent string
	hc        *http.Client
	l         *zap.Logger
}

func getToken(log *zap.Logger) (string, error) {
	token, ok := os.LookupEnv("YT_TOKEN")
	if ok {
		log.Debugf("Use YT_TOKEN")
		return token, nil
	}
	u, err := user.Current()
	if err == nil {
		data, err := ioutil.ReadFile(filepath.Join(u.HomeDir, ".yql/token"))
		if err == nil {
			log.Debugf("Use .yql/token")
			token = strings.Trim(string(data), "\n")
			return token, nil
		}
	}
	token, err = oauth.GetTokenBySSH(
		context.TODO(),
		clientID, clientSecret)
	if err != nil {
		return "", fmt.Errorf("fail to exchange ssh key to oauth token: %w", err)
	}
	log.Debugf("Use ssh-key token")
	return token, nil
}

func NewClient(l *zap.Logger) (*Client, error) {
	token, err := getToken(l)
	if err != nil {
		return nil, err
	}
	return &Client{
		baseURL:   baseURL,
		token:     token,
		userAgent: "coronerctl",
		hc:        &http.Client{Timeout: time.Second * 10},
		l:         l,
	}, nil
}

func (c *Client) Query(e Engine, db, q string) *Query {
	return &Query{
		c:      c,
		url:    c.baseURL + "/operations",
		DB:     db,
		Engine: e,
		Query:  q,
		l:      c.l,
	}
}

func (c *Client) doReq(t string, url string, args url.Values, data io.Reader) (*fastjson.Value, error) {
	var (
		resp *http.Response
		err  error
	)
	req, err := http.NewRequest(t, url, data)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Accept", "application/json")
	req.Header.Set("User-Agent", c.userAgent)
	req.Header.Set("Authorization", "OAuth "+c.token)
	req.URL.RawQuery = args.Encode()
	c.l.Debugf("request %s %s\n", req.Method, req.URL)

	for try := 1; try <= retries; try++ {
		resp, err = c.hc.Do(req)
		if os.IsTimeout(err) {
			c.l.Warnf("attempt: %d, %s", try, err)
			continue
		} else {
			break
		}
	}
	if err != nil {
		c.l.Errorf("Error with connection: %s", err)
		return nil, err
	}

	defer resp.Body.Close()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		c.l.Errorf("Error with reading: %s", err)
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		return nil, &HTTPErr{
			Code: resp.StatusCode,
			Text: string(body),
		}
	}
	d, err := fastjson.ParseBytes(body)
	if err != nil {
		return nil, err
	}
	if len(d.GetArray("errors")) > 0 {
		return nil, fmt.Errorf("%s", d.GetArray("errors")[0].GetStringBytes("message"))
	}
	return d, nil
}

type Query struct {
	c      *Client
	url    string
	ID     string
	DB     string
	Engine Engine
	Query  string
	Status Status
	Data   []*fastjson.Value
	Schema []*fastjson.Value
	l      *zap.Logger
}

type queryRequest struct {
	DB      string `json:"database"`
	Content string `json:"content"`
	Action  string `json:"action"`
	Type    Engine `json:"type"`
}

func (q *Query) Run(action string) error {
	if err := q.Start(action); err != nil {
		return err
	}
	for q.Status != StatusCompleted && q.Status != StatusAborted && q.Status != StatusError {
		if err := q.updateStatus(); err != nil {
			q.l.Errorf("%s", err)
		}
		time.Sleep(time.Second)
	}
	return q.updateData()
}

func (q *Query) Start(action string) error {
	data, err := json.Marshal(&queryRequest{
		DB:      q.DB,
		Content: q.Query,
		Action:  action,
		Type:    q.Engine,
	})
	if err != nil {
		return fmt.Errorf("unable to serialize request: %w", err)
	}
	resp, err := q.c.doReq(http.MethodPost, q.url, nil, bytes.NewReader(data))
	if err != nil {
		return err
	}
	q.ID = string(resp.GetStringBytes("id"))
	q.Status = Status(resp.GetStringBytes("status"))
	q.l.Infof("created operation https://yql.yandex-team.ru/Operations/%s\n", q.ID)
	return nil
}

func (q *Query) updateStatus() error {
	resp, err := q.c.doReq(http.MethodGet, q.url+"/"+q.ID, nil, nil)
	if err != nil {
		return err
	}
	q.l.Debugf("status:%s\n", q.Status)
	q.Status = Status(resp.GetStringBytes("status"))
	return nil
}

func (q *Query) updateData() error {
	args := url.Values{}
	args.Set("filters", "DATA")
	resp, err := q.c.doReq(http.MethodGet, q.url+"/"+q.ID+"/results", args, nil)
	if err != nil {
		return err
	}
	q.Data = resp.GetArray("data")[0].GetArray("Write")
	return nil
}
