
from argparse import ArgumentParser, ArgumentTypeError
import requests
import json
import logging
import os
import threading
import nirvana.job_context as nv
from infra.rsm.coroner.scripts.extract_cpu_host_walle_project.walle_model import compute_walle_stat
from yp.client import YpClient, find_token
from yp.common import YP_PRODUCTION_CLUSTERS
from yt.yson import YsonEntity


def extract_cpu_model(fqdn):
    unknown_service = "UNKNOWN"

    inv = requests.get("https://api.wall-e.yandex-team.ru/v1/hosts/{0}".format(fqdn)).json().get("inv", {})
    data = requests.get("https://bot.yandex-team.ru/api/consistof.php?inv={0}&format=json".format(str(inv))).json()

    try:
        if "YANDEX" not in data["data"]["item_segment2"]:
            return data["data"]["item_segment2"]
        else:
            for component in data["data"]["Components"]:
                if component["item_segment3"] == "CPU":
                    return component["item_segment1"]

            logging.info("Could not process cpu model for {0}".format(fqdn))
            return unknown_service
    except KeyError:
        return


def str2bool(v):
    if isinstance(v, bool):
        return v
    if v.lower() in ('yes', 'true', 't', 'y', '1'):
        return True
    elif v.lower() in ('no', 'false', 'f', 'n', '0'):
        return False
    else:
        raise ArgumentTypeError('Boolean value expected.')


def process_fqdns(fqdns, fqdn_host_mapping, result):

    for fqdn in fqdns:
        processed_fqdn = {
            "fqdn": fqdn, "walle_project": fqdn_host_mapping[fqdn].project,
            "cpu_model": extract_cpu_model(fqdn_host_mapping[fqdn].fqdn),
            "dc": fqdn_host_mapping[fqdn].location
        }
        result.append(processed_fqdn)


def split_chunks(lst, chnksize):

    for i in range(0, len(lst), chnksize):
        yield lst[i:i + chnksize]


def parallelize_data_extraction(fqdns, walle_hosts):

    splitted_fqdns = list(split_chunks(fqdns, int(len(fqdns) / 5)))
    result, threads = [], []

    for i in range(len(splitted_fqdns)):
        t = threading.Thread(target=process_fqdns, args=(splitted_fqdns[i], walle_hosts, result))
        t.start()
        threads.append(t)

    for thread in threads:
        thread.join()

    return result


def extract_mapping_yp(resource, first_path, second_path, filter=None):

    host_cpu_mapping = {}

    for dc in YP_PRODUCTION_CLUSTERS:
        client = YpClient(dc, config=dict(token=find_token()))

        if filter:
            dc_mapping = {i[1]["value"]: i[0]["value"] for i in
                client.select_objects(
                resource, selectors=[first_path, second_path], filter=filter, enable_structured_response=True
            )["results"]}
        else:
            dc_mapping = {i[1]["value"]: i[0]["value"] for i in
                          client.select_objects(
                              resource, selectors=[first_path, second_path], enable_structured_response=True
                          )["results"]}

        host_cpu_mapping = {**host_cpu_mapping, **dc_mapping}

    return host_cpu_mapping


def process_yp_field(record, field, mapping, substitute_value=None):

    if record["fqdn"] in mapping and mapping[record["fqdn"]] != YsonEntity():
        record[field] = mapping[record["fqdn"]]
    else:
        record[field] = substitute_value


if __name__ == '__main__':
    parser = ArgumentParser()
    parser.add_argument("--local_run", type=str2bool, default=False)
    args = parser.parse_args()

    data = compute_walle_stat().host_map
    hosts = list(data.keys())

    data_extracted = parallelize_data_extraction(hosts, data)
    yp_extracted_data_fqdn_cpu = extract_mapping_yp("node", "/labels/cpu_model", "/meta/id")
    yp_extracted_data_vcpu_fqdn = extract_mapping_yp(
        "resource", "/spec/cpu/cpu_to_vcpu_factor", "/meta/node_id", filter="[/meta/kind]='cpu'"
    )
    yp_extracted_data_fqdn_segment = extract_mapping_yp("node", "/labels/segment", "/meta/id")

    for record in data_extracted:
        process_yp_field(record, "yp_cpu", yp_extracted_data_fqdn_cpu)
        process_yp_field(record, "yp_vcpu_factor", yp_extracted_data_vcpu_fqdn, 1.0)
        process_yp_field(record, "yp_segment", yp_extracted_data_fqdn_segment, "OUT_OF_YP")

    if args.local_run is False:
        job_context = nv.context()
        outputs = job_context.get_outputs()
        filename = os.environ.get("OUTPUT_FILE")

        with open(outputs.get(filename), "w") as write_file:
            json.dump(data_extracted, write_file)
    else:
        json.dump(data_extracted, open("extracted_coroner_data.json", "w"))
