import logging
import pickle
import time
from collections import defaultdict

from walle_api import WalleClient
from walle_api.client import WalleConnectionError

WALLE_CACHE_VERSION = 1
WALLE_CACHE_TTL = 720 * 60 * 60


class WalleHost(object):
    __slots__ = ["fqdn", "project", "location"]

    def __init__(self):
        self.fqdn = None
        self.project = None
        self.location = None


class WalleProject(object):
    __slots__ = ["id", "tags"]

    def __init__(self):
        self.id = None
        self.tags = []


def compute_stats(host_map, project_map):
    client = WalleClient()

    while True:
        try:
            for host in client.iter_hosts(fields=["name", "project", "location.short_datacenter_name"]):
                if not host.get("name"):
                    continue

                walle_host = host_map[host["name"]]
                walle_host.fqdn = host["name"]
                walle_host.project = host["project"]
                walle_host.location = host["location"]["short_datacenter_name"]
            break
        except WalleConnectionError:
            time.sleep(30)

    for project in client.get_projects(fields=["id", "tags"])["result"]:
        walle_project = project_map[project["id"]]
        walle_project.id = project["id"]
        walle_project.tags = project.get("tags", [])


class WalleStat(object):
    def __init__(self):
        self.host_map = defaultdict(WalleHost)
        self.project_map = defaultdict(WalleProject)
        self._version = WALLE_CACHE_VERSION
        self._timestamp = time.time()

    def freeze(self):
        self.host_map = dict(self.host_map)

    def check_actual(self):
        if self._version != WALLE_CACHE_VERSION:
            raise Exception("version mismatch")
        if time.time() - self._timestamp > WALLE_CACHE_TTL:
            raise Exception("cache expired")


def compute_walle_stat():
    stat = WalleStat()
    logging.info("Get walle stats")
    compute_stats(stat.host_map, stat.project_map)
    stat.freeze()
    return stat


def get_walle_stat_cached():
    """
    :rtype: WalleStat
    """
    file_name = "walle_stat.tmp"
    try:
        with open(file_name, "rb") as stream:
            result = pickle.load(stream)  # type: WalleStat
            result.check_actual()
            return result
    except Exception as e:
        print(e)
        result = compute_walle_stat()
        result.check_actual()
        with open(file_name, "wb") as stream:
            pickle.dump(result, stream, protocol=pickle.HIGHEST_PROTOCOL)
        return result
