package cmd

import (
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"github.com/spf13/cobra"
	jaegercfg "github.com/uber/jaeger-client-go/config"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/diskmanager"
	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/library/go/core/buildinfo"
)

var (
	cfgFile         string
	debug           bool
	serviceEndpoint string
	tracerEndpoint  string
	tracerEnable    bool
)

// rootCmd represents the base command when called without any subcommands
var rootCmd = &cobra.Command{
	Use:   "diskmanager",
	Short: "Diskmanager service",
	Long: `Diskmanager service
    Wiki: https://wiki.yandex-team.ru/runtime-cloud/disk-manager
    Docs: https://a.yandex-team.ru/arc/trunk/arcadia/infra/diskmanager/README.md`,
	Run: func(cmd *cobra.Command, args []string) {
		doMain(cmd, args)
	},
}

// Execute adds all child commands to the root command and sets flags appropriately.
// This is called by main.main(). It only needs to happen once to the rootCmd.
func Execute() {
	if err := rootCmd.Execute(); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}

func init() {
	if buildinfo.Info.ProgramVersion == "" {
		rootCmd.Version = "N/A"
	} else {
		rootCmd.Version = buildinfo.Info.ProgramVersion
	}

	//cobra.OnInitialize(initConfig)
	rootCmd.PersistentFlags().BoolVarP(&debug, "debug", "d", false, "enable debug level")
	rootCmd.PersistentFlags().StringVar(&cfgFile, "config", "/etc/diskmanager", "config file (default is /etc/diskmanager.yaml)")
	rootCmd.PersistentFlags().StringVarP(&serviceEndpoint, "service", "S", diskmanager.DefaultServerAddress, "service GRPC endpoint")

	rootCmd.PersistentFlags().BoolVar(&tracerEnable, "enable-opentrace", false, "enable opentrace")
	rootCmd.PersistentFlags().StringVar(&tracerEndpoint, "opentrace-endpoint", "", "opentrace endpoint url")

}

// initConfig reads in config file and ENV variables if set.
//func initConfig() {
//}

func doMain(cmd *cobra.Command, args []string) {
	ilog.Init()
	ll := ilog.Log()
	defer ll.Sync()

	if debug {
		ilog.Cfg().Level.SetLevel(zap.DebugLevel)
	}

	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM)
	if tracerEnable {
		ll.Info("Init opentracer")

		closer, err := diskmanager.InitTracer(ll, jaegercfg.Configuration{})
		if err != nil {
			ll.Fatal("Fail to initalize tracer", zap.Error(err))
			os.Exit(1)
		}
		defer closer.Close()
	}

	cfg := diskmanager.DefaultConfiguration
	cfg.GrpcEndpoint = serviceEndpoint
	dm, err := cfg.NewDiskmanager()
	if err != nil {
		ll.Fatal("Fail to create Diskmanager", zap.Error(err))
		os.Exit(1)
	}

	go func() {
		<-sigs
		dm.Stop("stopping by signal")
	}()

	if err := dm.Serve(); err != nil {
		ll.Fatal("failed to serve", zap.Error(err))
		os.Exit(1)
	}

}
