package cmd

import (
	"fmt"
	"os"

	"github.com/spf13/cobra"
	"go.uber.org/zap"

	dmclient "a.yandex-team.ru/infra/rsm/diskmanager/internal/client"
	"a.yandex-team.ru/infra/rsm/diskmanager/internal/diskmanager"
	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/library/go/core/buildinfo"
)

var (
	cfgFile         string
	debug           bool
	serviceEndpoint string
	tracerEndpoint  string
	tracerEnable    bool
	dumpJSON        bool
	client          *dmclient.Client
)

// rootCmd represents the base command when called without any subcommands
var rootCmd = &cobra.Command{
	Use:   "dmctl",
	Short: "Diskmanager service client",
	Long: `Diskmanager service client
        Wiki: https://wiki.yandex-team.ru/runtime-cloud/disk-manager
        Docs: https://a.yandex-team.ru/arc/trunk/arcadia/infra/diskmanager/README.md`,
	PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
		return initClient()
	},
	PersistentPostRunE: func(cmd *cobra.Command, args []string) error {
		return destroyClient()
	},
	Version: buildinfo.Info.ProgramVersion,
}

// Execute adds all child commands to the root command and sets flags appropriately.
// This is called by main.main(). It only needs to happen once to the rootCmd.
func Execute() {
	if err := rootCmd.Execute(); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}

func init() {
	if buildinfo.Info.ProgramVersion == "" {
		rootCmd.Version = "N/A"
	} else {
		rootCmd.Version = buildinfo.Info.ProgramVersion
	}

	//cobra.OnInitialize(initConfig)
	rootCmd.PersistentFlags().BoolVarP(&debug, "debug", "d", false, "enable debug level")
	rootCmd.PersistentFlags().StringVar(&cfgFile, "config", "/etc/diskmanager", "config file (default is /etc/diskmanager.yaml)")
	rootCmd.PersistentFlags().StringVarP(&serviceEndpoint, "service", "S", diskmanager.DefaultServerAddress, "service GRPC endpoint")

	rootCmd.PersistentFlags().BoolVar(&tracerEnable, "enable-opentrace", false, "enable opentrace")
	rootCmd.PersistentFlags().StringVar(&tracerEndpoint, "opentrace-endpoint", "", "opentrace endpoint url")
}

func initClient() error {
	var err error
	ilog.Init()
	ll := ilog.Log()
	defer ll.Sync()
	ilog.Cfg().Level.SetLevel(zap.WarnLevel)
	if debug {
		ilog.Cfg().Level.SetLevel(zap.DebugLevel)
	}
	client, err = dmclient.NewClient(ll, serviceEndpoint)
	if err != nil {
		return err
	}
	return nil
}

func destroyClient() error {
	ilog.Log().Debug("destroy client", zap.Any("client", client))
	client.Close()
	// If log is os.Stdout it has not sync method, just ignore error
	_ = ilog.Log().Sync()
	return nil
}
