package client

import (
	"io"

	grpc_opentracing "github.com/grpc-ecosystem/go-grpc-middleware/tracing/opentracing"
	"github.com/opentracing/opentracing-go"
	"go.uber.org/zap"
	"google.golang.org/grpc"
	_ "google.golang.org/grpc/metadata"

	jaeger "github.com/uber/jaeger-client-go"
	jaegercfg "github.com/uber/jaeger-client-go/config"
	jaegerlog "github.com/uber/jaeger-client-go/log"

	pb "a.yandex-team.ru/infra/rsm/diskmanager/api"
)

type Client struct {
	Client pb.DiskManagerClient
	Log    *zap.Logger
	conn   *grpc.ClientConn
}

func InitTracer(logger *zap.Logger, cfg jaegercfg.Configuration) (io.Closer, error) {

	if cfg.ServiceName == "" {
		cfg = jaegercfg.Configuration{
			ServiceName: "dmctl",
			Sampler: &jaegercfg.SamplerConfig{
				Type:  jaeger.SamplerTypeConst,
				Param: 1,
			},
			Reporter: &jaegercfg.ReporterConfig{
				CollectorEndpoint: "http://dmws02.man.yp-c.yandex.net:14268/api/traces?format=jaeger.thrift",
				LogSpans:          true,
			},
		}
		logger.Debug("Generate tracer", zap.Any("config", cfg))
	}
	jLogger := jaegerlog.StdLogger
	tracer, closer, err := cfg.NewTracer(
		jaegercfg.Logger(jLogger),
	)
	if err != nil {
		return nil, err
	}
	logger.Info("init tracer", zap.String("sevice", cfg.ServiceName))
	opentracing.SetGlobalTracer(tracer)
	return closer, nil
}

func NewClient(l *zap.Logger, ep string) (*Client, error) {
	l.Debug("NewClient", zap.Any("endpoint", ep))
	conn, err := grpc.Dial(ep, grpc.WithInsecure(),
		grpc.WithUnaryInterceptor(grpc_opentracing.UnaryClientInterceptor()),
		grpc.WithStreamInterceptor(grpc_opentracing.StreamClientInterceptor()),
	)
	if err != nil {
		l.Fatal("fail to conect", zap.Error(err))
		return nil, err
	}
	cl := &Client{
		conn:   conn,
		Log:    l,
		Client: pb.NewDiskManagerClient(conn),
	}
	return cl, nil
}

func (dc *Client) Close() {
	if dc != nil {
		err := dc.conn.Close()
		if err != nil {
			dc.Log.Error("close failed", zap.Error(err))
		}
	}
}
